package ru.yandex.solomon.slog;

import javax.annotation.WillNotClose;

import io.netty.buffer.ByteBuf;

import ru.yandex.monlib.metrics.encode.spack.StringPool;
import ru.yandex.monlib.metrics.encode.spack.format.MetricTypes;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.slog.compression.DecodeStream;
import ru.yandex.solomon.util.collection.RecyclableLong2ObjectMap;

/**
 * @author Vladimir Gordiychuk
 */
public class ResolvedLogMetricsIteratorImpl implements ResolvedLogMetricsIterator {
    private final ResolvedLogMetricsHeader header;
    private final DecodeStream in;
    private final RecyclableLong2ObjectMap<Label> labelCache;
    private LabelsBuilder labelsBuilder;
    private StringPool keysPool;
    private StringPool valuesPool;
    private LabelAllocator allocator;
    private int pos;

    public ResolvedLogMetricsIteratorImpl(@WillNotClose ByteBuf in) {
        this(in, Labels.allocator);
    }

    public ResolvedLogMetricsIteratorImpl(@WillNotClose ByteBuf in, LabelAllocator allocator) {
        this(new ResolvedLogMetricsHeader(in), in, allocator);
    }

    public ResolvedLogMetricsIteratorImpl(ResolvedLogMetricsHeader header, @WillNotClose ByteBuf in, LabelAllocator allocator) {
        this.header = header;
        this.allocator = allocator;
        this.in = DecodeStream.create(header.compressionAlg, in.retain());
        this.labelCache = RecyclableLong2ObjectMap.newInstance();
        this.labelsBuilder = Labels.builder(16, allocator);
        readStringPools();
    }

    private void readStringPools() {
        final int maxPoolSize = Math.max(header.labelNamesSize, header.labelValuesSize);
        final byte[] poolBuffer = new byte[maxPoolSize];

        in.read(poolBuffer, 0, header.labelNamesSize);
        keysPool = new StringPool(poolBuffer, header.labelNamesSize);

        in.read(poolBuffer, 0, header.labelValuesSize);
        valuesPool = new StringPool(poolBuffer, header.labelValuesSize);
    }

    private void readLabels(LabelsBuilder builder) {
        final int count = in.readVarint32();
        for (int i = 0; i < count; i++) {
            final int nameIdx = in.readVarint32();
            final int valueIdx = in.readVarint32();

            final long cacheIdx = (long) nameIdx << 32 | (long) valueIdx;
            Label label = labelCache.get(cacheIdx);
            if (label == null) {
                String key = keysPool.get(nameIdx);
                String value = valuesPool.get(valueIdx);
                label = allocator.alloc(key, value);
                labelCache.put(cacheIdx, label);
            }
            builder.add(label);
        }
    }

    @Override
    public boolean next(ResolvedLogMetricsRecord record) {
        if (pos++ >= header.metricsCount) {
            return false;
        }

        labelsBuilder.clear();
        record.type = MetricTypes.metricType(in.readByte());
        readLabels(labelsBuilder);
        record.labels = labelsBuilder.build();
        record.shardId = in.readVarint32();
        record.localId = in.readLongLe();
        return true;
    }

    @Override
    public void close() {
        in.close();
        labelCache.recycle();
    }
}
