package ru.yandex.solomon.slog;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;

import ru.yandex.misc.lang.ShortUtils;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.MetricTypes;
import ru.yandex.monlib.metrics.encode.spack.format.MetricValuesType;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.bits.BitArray;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.slog.compression.EncodeStream;

/**
 * @author Vladimir Gordiychuk
 */
public class SnapshotLogDataBuilderImpl implements SnapshotLogDataBuilder {
    private final SnapshotLogDataHeader header;

    private final EncodeStream encoder;
    private boolean build;

    public SnapshotLogDataBuilderImpl(CompressionAlg alg, int numId, ByteBufAllocator allocator) {
        this.header = new SnapshotLogDataHeader(alg, numId);
        this.encoder = EncodeStream.create(alg, allocator);
        this.encoder.writeHeader(buf -> buf.writeZero(header.size()));
    }

    @Override
    public int onTimeSeries(MetricArchiveImmutable archive) {
        header.metricsCount++;
        header.pointsCount += archive.getRecordCount();
        int size = 0;
        size += writeType(archive);
        size += writeColumnSetMask(archive);
        size += writeContent(archive);
        return size;
    }

    @Override
    public ByteBuf build() {
        if (build) {
            throw new IllegalStateException("Already build");
        }

        var buffer = encoder.finish();
        try {
            int idx = buffer.writerIndex();
            header.writeTo(buffer.resetWriterIndex());
            buffer.writerIndex(idx);
            build = true;
        } catch (Throwable e) {
            buffer.release();
            throw new RuntimeException(e);
        }
        return buffer;
    }

    @Override
    public void close() {
        encoder.close();
    }

    private int writeType(MetricArchiveImmutable archive) {
        encoder.writeByte(MetricTypes.pack(MetricTypeConverter.fromProto(archive.getType()), MetricValuesType.NONE));
        return 1;
    }

    private int writeColumnSetMask(MetricArchiveImmutable archive) {
        encoder.writeShortLe(ShortUtils.toShortExact(archive.columnSetMask()));
        return 2;
    }

    private int writeContent(MetricArchiveImmutable archive) {
        BitBuf compressed = archive.getCompressedDataRaw();
        long bits = compressed.readableBits();
        int bytes = BitArray.arrayLengthForBits(bits);
        encoder.writeLongLe(bits);
        encoder.write(compressed.array(), compressed.arrayOffset(), bytes);
        return 8 + bytes;
    }
}
