package ru.yandex.solomon.slog;

import io.netty.buffer.ByteBuf;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;

/**
 * @author Vladimir Gordiychuk
 */
public class SnapshotLogDataHeader {
    /**
     * defines how many bytes used for header by current implementation
     */
    private static final short HEADER_SIZE = 16;

    /**
     * expected format magic number
     */
    public static final short VALID_MAGIC = 0x4453; // "SD" in LE-order

    public final SnapshotLogDataVersion version;
    public final CompressionAlg compressionAlg;
    public final int numId;
    public int metricsCount;
    public int pointsCount;

    public SnapshotLogDataHeader(CompressionAlg compressionAlg, int numId) {
        this(compressionAlg, numId, 0, 0);
    }

    public SnapshotLogDataHeader(CompressionAlg compressionAlg, int numId, int metrics, int points) {
        this.version = SnapshotLogDataVersion.CURRENT;
        this.compressionAlg = compressionAlg;
        this.numId = numId;
        this.metricsCount = metrics;
        this.pointsCount = points;
    }


    public SnapshotLogDataHeader(ByteBuf in) {
        if (in.readableBytes() < HEADER_SIZE) {
            throw new CorruptedBinaryDataRuntimeException(
                "not enough bytes in buffer to read header, " +
                    "need at least: " + HEADER_SIZE + ", but got: " + in.readableBytes());
        }

        final int magic = in.readShortLE();
        if (magic != VALID_MAGIC) {
            throw new CorruptedBinaryDataRuntimeException("invalid magic, expected " +
                Integer.toString(VALID_MAGIC, 16) + ", got " + Integer.toString(magic, 16));
        }

        this.version = SnapshotLogDataVersion.valueOf(in.readByte());
        this.compressionAlg = CompressionAlg.valueOf(in.readByte());
        this.numId = in.readIntLE();
        this.metricsCount = in.readIntLE();
        this.pointsCount = in.readIntLE();
    }

    public void writeTo(ByteBuf buffer) {
        writeTo(buffer, metricsCount, pointsCount);
    }

    public void writeTo(ByteBuf buffer, int metricsCount, int pointsCount) {
        buffer
            .writeShortLE(VALID_MAGIC) // 2
            .writeByte(version.number) // 3
            .writeByte(compressionAlg.value()) // 4
            .writeIntLE(numId) // 8
            .writeIntLE(metricsCount) // 12
            .writeIntLE(pointsCount); // 16
    }

    public void setMetricsCount(int idx, ByteBuf buffer, int count) {
        buffer.setIntLE(idx + 8, count);
    }

    public void setPointsCount(int idx, ByteBuf buffer, int count) {
        buffer.setIntLE(idx + 12, count);
    }

    public int size() {
        return HEADER_SIZE;
    }

    @Override
    public String toString() {
        return "SnapshotLogDataHeader{" +
            "version=" + version +
            ", compressionAlg=" + compressionAlg +
            ", numId=" + Integer.toUnsignedLong(numId) +
            ", metricsCount=" + metricsCount +
            ", pointsCount=" + pointsCount +
            '}';
    }
}
