package ru.yandex.solomon.slog;

import java.util.NoSuchElementException;

import javax.annotation.WillNotClose;

import io.netty.buffer.ByteBuf;

import ru.yandex.monlib.metrics.encode.spack.format.MetricTypes;
import ru.yandex.solomon.codec.BinaryAggrGraphDataListIterator;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.bits.BitArray;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.slog.compression.DecodeStream;

/**
 * @author Vladimir Gordiychuk
 */
public class SnapshotLogDataIteratorImpl implements LogDataIterator {
    private final SnapshotLogDataHeader header;
    private final DecodeStream in;
    private int pos;
    private byte[] tmpBuffer;

    public SnapshotLogDataIteratorImpl(SnapshotLogDataHeader header, @WillNotClose ByteBuf buffer) {
        this.header = header;
        this.in = DecodeStream.create(header.compressionAlg, buffer.retain());
        this.tmpBuffer = new byte[0];
    }

    @Override
    public int getNumId() {
        return header.numId;
    }

    @Override
    public int getPointsCount() {
        return header.pointsCount;
    }

    @Override
    public int getMetricsCount() {
        return header.metricsCount;
    }

    @Override
    public boolean hasNext() {
        return pos < header.metricsCount;
    }

    @Override
    public void next(MetricArchiveMutable archive) {
        if (pos++ >= header.metricsCount) {
            throw new NoSuchElementException();
        }

        var metricType = readMetricType();
        int columnMask = readColumnMask();
        var compressed = readContent();

        // TODO: avoid repack for framed archives
        archive.setOwnerShardId(header.numId);
        archive.setType(metricType);
        archive.ensureBytesCapacity(columnMask, compressed.bytesSize());
        archive.addAllFrom(new BinaryAggrGraphDataListIterator(metricType, columnMask, compressed, 1));
    }

    private MetricType readMetricType() {
        return MetricTypeConverter.toNotNullProto(MetricTypes.metricType(in.readByte()));
    }

    private int readColumnMask() {
        return in.readShortLe();
    }

    private BitBuf readContent() {
        long bits = in.readLongLe();
        int bytes = BitArray.arrayLengthForBits(bits);
        if (tmpBuffer.length < bytes) {
            tmpBuffer = new byte[bytes];
        }

        in.read(tmpBuffer,0, bytes);
        return new HeapBitBuf(tmpBuffer, bits);
    }

    @Override
    public void close() {
        in.close();
    }
}
