package ru.yandex.solomon.slog;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.CompositeByteBuf;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.MetricTypes;
import ru.yandex.monlib.metrics.encode.spack.format.MetricValuesType;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;
import ru.yandex.solomon.slog.compression.EncodeStream;

/**
 * @author Vladimir Gordiychuk
 */
public class UnresolvedLogMetaBuilderImpl implements UnresolvedLogMetaBuilder {
    private final ByteBufAllocator allocator;
    private final CompressionAlg alg;

    private final int numId;
    private int metrics;
    private int points;
    private boolean hasCommonLabels;

    private EncodeStream meta;
    private StringPoolEncoder valuesPool;
    private StringPoolEncoder keysPool;
    private boolean build;

    public UnresolvedLogMetaBuilderImpl(int numId, CompressionAlg alg, ByteBufAllocator allocator) {
        this.allocator = allocator;
        this.alg = alg;
        this.numId = numId;
        try {
            this.meta = EncodeStream.create(alg, allocator);
            this.keysPool = new StringPoolEncoder(alg, allocator);
            this.valuesPool = new StringPoolEncoder(alg, allocator);
        } catch (Throwable e) {
            if (meta != null) {
                meta.close();
            }
            if (keysPool != null) {
                keysPool.close();
            }
            throw new RuntimeException(e);
        }
    }

    @Override
    public void onCommonLabels(Labels labels) {
        if (hasCommonLabels) {
            throw new CorruptedBinaryDataRuntimeException("common label shout be set before metrics");
        }
        writeLabels(labels);
        hasCommonLabels = true;
    }

    @Override
    public void onMetric(MetricType type, Labels labels, int points, int byteSize) {
        this.metrics++;
        this.points+=points;
        this.meta.writeByte(MetricTypes.pack(type, MetricValuesType.NONE));
        this.writeLabels(labels);
        this.meta.writeVarint32(points);
        this.meta.writeVarint32(byteSize);
    }

    @Override
    public ByteBuf build() {
        if (!hasCommonLabels) {
            throw new CorruptedBinaryDataRuntimeException("Not specified common labels");
        }
        if (build) {
            throw new IllegalStateException("Already build");
        }

        var keysPool = this.keysPool.finish();
        var valuesPool = this.valuesPool.finish();
        var meta = this.meta.finishStream();

        var header = new UnresolvedLogMetaHeader(
            numId,
            alg,
            this.keysPool.getBytesSize(),
            this.valuesPool.getBytesSize(),
            metrics,
            points);

        ByteBuf headerBuffer = allocator.buffer(header.size(), header.size());
        header.writeTo(headerBuffer);
        CompositeByteBuf compositeMeta = allocator.compositeBuffer();
        compositeMeta.addComponent(true, headerBuffer);
        compositeMeta.addComponent(true, keysPool);
        compositeMeta.addComponent(true, valuesPool);
        compositeMeta.addComponent(true, meta);
        build = true;
        return compositeMeta;
    }

    @Override
    public void close() {
        meta.close();
        keysPool.close();
        valuesPool.close();
    }

    private void writeLabels(Labels labels) {
        meta.writeVarint32(labels.size());
        labels.forEach(this::writeLabel);
    }

    private void writeLabel(Label label) {
        meta.writeVarint32(keysPool.writeString(label.getKey()));
        meta.writeVarint32(valuesPool.writeString(label.getValue()));
    }
}
