package ru.yandex.solomon.slog;

import io.netty.buffer.ByteBuf;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;

/**
 * @author Vladimir Gordiychuk
 */
public class UnresolvedLogMetaHeader {
    /**
     * defines how many bytes used for header by current implementation
     */
    private static final short HEADER_SIZE = 28;
    private static final short SKIP_BYTES = 3;

    /**
     * expected format magic number
     */
    private static final short VALID_MAGIC = 0x4D55; // "UM" in LE-order

    public final UnresolvedLogMetaVersion version;
    public final int numId;
    public final CompressionAlg compressionAlg;
    public final int labelNamesSize;
    public final int labelValuesSize;
    public final int metricsCount;
    public final int pointsCount;

    public UnresolvedLogMetaHeader(
        int numId,
        CompressionAlg compressionAlg,
        int labelNamesSize,
        int labelValuesSize,
        int metricsCount,
        int pointsCount)
    {
        this.version = UnresolvedLogMetaVersion.CURRENT;
        this.numId = numId;
        this.compressionAlg = compressionAlg;
        this.labelNamesSize = labelNamesSize;
        this.labelValuesSize = labelValuesSize;
        this.metricsCount = metricsCount;
        this.pointsCount = pointsCount;
    }

    public UnresolvedLogMetaHeader(ByteBuf in) {
        if (in.readableBytes() < HEADER_SIZE) {
            throw new CorruptedBinaryDataRuntimeException(
                "not enough bytes in buffer to read header, " +
                    "need at least: " + HEADER_SIZE + ", but got: " + in.readableBytes());
        }

        final int magic = in.readShortLE();
        if (magic != VALID_MAGIC) {
            throw new CorruptedBinaryDataRuntimeException("invalid magic, expected " +
                Integer.toString(VALID_MAGIC, 16) + ", got " + Integer.toString(magic, 16));
        }

        in.skipBytes(1);
        this.version = UnresolvedLogMetaVersion.valueOf(in.readByte());
        this.numId = in.readIntLE();
        this.labelNamesSize = in.readIntLE();
        this.labelValuesSize = in.readIntLE();
        this.metricsCount = in.readIntLE();
        this.pointsCount = in.readIntLE();
        this.compressionAlg = CompressionAlg.valueOf(in.readByte());
        in.skipBytes(SKIP_BYTES);
    }

    public void writeTo(ByteBuf buffer) {
        buffer
            .writeShortLE(VALID_MAGIC) // 2
            .writeZero(1) // 3
            .writeByte(version.number) // 4
            .writeIntLE(numId) // 8
            .writeIntLE(labelNamesSize) // 12
            .writeIntLE(labelValuesSize) // 16
            .writeIntLE(metricsCount) // 20
            .writeIntLE(pointsCount) // 24
            .writeByte(compressionAlg.value()) // 25
            .writeZero(SKIP_BYTES); // 28
    }

    public int size() {
        return HEADER_SIZE;
    }
}
