package ru.yandex.solomon.slog;

import javax.annotation.WillNotClose;

import io.netty.buffer.ByteBuf;

import ru.yandex.monlib.metrics.encode.spack.StringPool;
import ru.yandex.monlib.metrics.encode.spack.format.MetricTypes;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.slog.compression.DecodeStream;
import ru.yandex.solomon.util.collection.RecyclableLong2ObjectMap;

/**
 * @author Vladimir Gordiychuk
 */
public class UnresolvedLogMetaIteratorImpl implements UnresolvedLogMetaIterator {
    private final UnresolvedLogMetaHeader header;
    private final DecodeStream in;
    private final RecyclableLong2ObjectMap<Label> labelCache;
    private LabelsBuilder labelsBuilder;
    private Labels commonLabels;
    private StringPool keysPool;
    private StringPool valuesPool;
    private LabelAllocator allocator;
    private int pos;

    public UnresolvedLogMetaIteratorImpl(@WillNotClose ByteBuf in) {
        this(in, Labels.allocator);
    }

    public UnresolvedLogMetaIteratorImpl(@WillNotClose ByteBuf in, LabelAllocator allocator) {
        this(new UnresolvedLogMetaHeader(in), in, allocator);
    }

    public UnresolvedLogMetaIteratorImpl(UnresolvedLogMetaHeader header, @WillNotClose ByteBuf in, LabelAllocator allocator) {
        this.header = header;
        this.allocator = allocator;
        this.in = DecodeStream.create(header.compressionAlg, in.retain());
        try {
            this.labelCache = RecyclableLong2ObjectMap.newInstance();
            this.labelsBuilder = Labels.builder(16, allocator);
            readStringPools();
            if (header.metricsCount > 0) {
                readLabels(labelsBuilder);
            }
            this.commonLabels = labelsBuilder.build();
        } catch (Throwable e) {
            this.in.close();
            throw new RuntimeException(e);
        }
    }

    private void readStringPools() {
        final int maxPoolSize = Math.max(header.labelNamesSize, header.labelValuesSize);
        final byte[] poolBuffer = new byte[maxPoolSize];

        in.read(poolBuffer, 0, header.labelNamesSize);
        keysPool = new StringPool(poolBuffer, header.labelNamesSize);

        in.read(poolBuffer, 0, header.labelValuesSize);
        valuesPool = new StringPool(poolBuffer, header.labelValuesSize);
    }

    private void readLabels(LabelsBuilder builder) {
        final int count = in.readVarint32();
        for (int i = 0; i < count; i++) {
            final int nameIdx = in.readVarint32();
            final int valueIdx = in.readVarint32();

            final long cacheIdx = (long) nameIdx << 32 | (long) valueIdx;
            Label label = labelCache.get(cacheIdx);
            if (label == null) {
                String key = keysPool.get(nameIdx);
                String value = valuesPool.get(valueIdx);
                label = allocator.alloc(key, value);
                labelCache.put(cacheIdx, label);
            }
            builder.add(label);
        }
    }

    @Override
    public boolean next(UnresolvedLogMetaRecord record) {
        if (pos++ >= header.metricsCount) {
            return false;
        }

        labelsBuilder.clear();
        labelsBuilder.addAll(commonLabels);
        record.type = MetricTypes.metricType(in.readByte());
        readLabels(labelsBuilder);
        record.labels = labelsBuilder.build();
        record.points = in.readVarint32();
        record.dataSize = in.readVarint32();
        return true;
    }

    @Override
    public void close() {
        in.close();
        labelCache.recycle();
    }
}
