package ru.yandex.solomon.slog.compression;

import java.nio.charset.StandardCharsets;

import javax.annotation.WillClose;

import io.netty.buffer.ByteBuf;

import ru.yandex.solomon.slog.compression.alg.Decompressor;

/**
 * @author Vladimir Gordiychuk
 */
public class CompressDecodeStream implements DecodeStream {
    private final ByteBuf uncompressed;
    private final ByteBuf compressed;
    private final Decompressor decompressor;

    public CompressDecodeStream(@WillClose ByteBuf compressed, Decompressor decompressor) {
        this.compressed = compressed;
        this.uncompressed = compressed.hasArray() ? compressed.alloc().heapBuffer() : compressed.alloc().directBuffer();
        this.decompressor = decompressor;
    }

    @Override
    public byte readByte() {
        ensureReadable(1);
        if (!uncompressed.isReadable()) {
            return -1;
        }
        return uncompressed.readByte();
    }

    @Override
    public int readIntLe() {
        ensureReadable(4);
        return uncompressed.readIntLE();
    }

    @Override
    public int readShortLe() {
        ensureReadable(2);
        return uncompressed.readShortLE();
    }

    @Override
    public long readLongLe() {
        ensureReadable(8);
        return uncompressed.readLongLE();
    }

    @Override
    public double readDoubleLe() { ;
        return Double.longBitsToDouble(readLongLe());
    }

    @Override
    public int readVarint32() {
        byte firstByte = readByte();
        if ((firstByte & 0x80) == 0) {
            return firstByte;
        }

        int result = firstByte & 0x7f;
        int offset = 7;
        for (; offset < 32; offset += 7) {
            final byte b = readByte();
            result |= (b & 0x7f) << offset;
            if ((b & 0x80) == 0) {
                return result;
            }
        }
        throw new RuntimeException("too many bytes for varint32");
    }

    @Override
    public String readString() {
        ensureReadable(1);
        int offset = 0;
        while (true) {
            int from = uncompressed.readerIndex();
            int index = uncompressed.indexOf(offset + from, uncompressed.writerIndex(), (byte) 0);
            if (index < 0) {
                // decode one more frame
                offset = uncompressed.readableBytes();
                decodeNextFrame();
                if (uncompressed.readableBytes() == 0) {
                    throw new RuntimeException("unable to read string, stream ended");
                }
            } else {
                String string = uncompressed.toString(from, index - from, StandardCharsets.UTF_8);
                uncompressed.readerIndex(index + 1);
                return string;
            }
        }
    }

    @Override
    public void read(byte[] buffer, int offset, int length) {
        while (length != 0) {
            int remaining = uncompressed.readableBytes();
            if (length < remaining) {
                uncompressed.readBytes(buffer, offset, length);
                return;
            }

            uncompressed.readBytes(buffer, offset, remaining);
            offset += remaining;
            length -= remaining;
            uncompressed.clear();
            decodeNextFrame();
        }
    }

    @Override
    public void read(ByteBuf buffer, int length) {
        buffer.ensureWritable(length);
        while (length != 0) {
            int remaining = uncompressed.readableBytes();
            if (length < remaining) {
                uncompressed.readBytes(buffer, length);
                return;
            }

            uncompressed.readBytes(buffer, remaining);
            length -= remaining;
            uncompressed.clear();
            decodeNextFrame();
        }
    }

    @Override
    public void skipBytes(int length) {
        while (length != 0) {
            int remaining = uncompressed.readableBytes();
            if (length < remaining) {
                uncompressed.skipBytes(length);
                return;
            }

            uncompressed.skipBytes(remaining);
            length -= remaining;
            uncompressed.clear();
            decodeNextFrame();
        }
    }

    private void ensureReadable(int bytes) {
        int readable = uncompressed.readableBytes();
        if (readable > bytes) {
            return;
        }

        if (readable == 0) {
            uncompressed.clear();
        } else {
            uncompressed.setBytes(0, uncompressed, uncompressed.readerIndex(), readable);
            uncompressed.resetReaderIndex();
            uncompressed.writerIndex(readable);
        }

        decodeNextFrame();
    }

    private void decodeNextFrame() {
        decompressor.decompress(compressed, uncompressed);
    }

    @Override
    public void close() {
        uncompressed.release();
        compressed.release();
    }
}
