package ru.yandex.solomon.slog.compression;

import java.util.function.Consumer;

import com.google.common.base.Throwables;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;

import ru.yandex.solomon.slog.compression.alg.Compressor;

/**
 * @author Vladimir Gordiychuk
 * @see ru.yandex.monlib.metrics.encode.spack.compression.FrameEncodeStream
 */
public class CompressEncodeStream extends UncompressedEncodeStream {
    private static final int DEFAULT_FRAME_SIZE = 64 << 10; // 64 KiB
    private final Compressor compressor;
    private final ByteBuf result;
    private boolean build;

    public CompressEncodeStream(ByteBufAllocator allocator, Compressor compressor) {
        super(allocator, 0, DEFAULT_FRAME_SIZE);
        this.compressor = compressor;
        try {
            this.result = allocator.buffer();
        } catch (Throwable e) {
            super.close();
            Throwables.throwIfUnchecked(e);
            throw new RuntimeException(e);
        }
    }

    @Override
    protected void doFlush(ByteBuf buffer) {
        compressor.compress(buffer, result);
        buffer.clear();
    }

    @Override
    public void writeHeader(Consumer<ByteBuf> writer) {
        writer.accept(result);
    }

    @Override
    public ByteBuf finish() {
        if (build) {
            throw new RuntimeException("Already build");
        }

        var buffer = super.finish();
        buffer.release();
        build = true;
        return result;
    }

    @Override
    public ByteBuf finishStream() {
        if (build) {
            throw new RuntimeException("Already build");
        }

        var buffer = super.finishStream();
        buffer.release();
        compressor.finish(result);
        build = true;
        return result;
    }

    @Override
    public void close() {
        if (build) {
            return;
        }
        super.close();
        result.release();
    }
}
