package ru.yandex.solomon.slog.compression;

import java.nio.charset.StandardCharsets;

import io.netty.buffer.ByteBuf;

/**
 * @author Vladimir Gordiychuk
 */
public class UncompressedDecodeStream implements DecodeStream {
    private final ByteBuf buffer;

    public UncompressedDecodeStream(ByteBuf buffer) {
        this.buffer = buffer;
    }

    @Override
    public byte readByte() {
        return buffer.readByte();
    }

    @Override
    public int readIntLe() {
        return buffer.readIntLE();
    }

    @Override
    public int readShortLe() {
        return buffer.readShortLE();
    }

    @Override
    public long readLongLe() {
        return buffer.readLongLE();
    }

    @Override
    public double readDoubleLe() {
        return Double.longBitsToDouble(readLongLe());
    }

    @Override
    public int readVarint32() {
        byte firstByte = readByte();
        if ((firstByte & 0x80) == 0) {
            return firstByte;
        }

        int result = firstByte & 0x7f;
        int offset = 7;
        for (; offset < 32; offset += 7) {
            final byte b = readByte();
            result |= (b & 0x7f) << offset;
            if ((b & 0x80) == 0) {
                return result;
            }
        }
        throw new RuntimeException("too many bytes for varint32");
    }

    @Override
    public String readString() {
        int from = buffer.readerIndex();
        int index = buffer.indexOf(from, buffer.writerIndex(), (byte) 0);
        if (index < 0) {
            throw new RuntimeException("Zero bytes for string not found");
        }
        String string = buffer.toString(from, index - from, StandardCharsets.UTF_8);
        buffer.readerIndex(index + 1);
        return string;
    }

    @Override
    public void read(byte[] buffer, int offset, int length) {
        this.buffer.readBytes(buffer, offset, length);
    }

    @Override
    public void read(ByteBuf buffer, int length) {
        buffer.ensureWritable(length);
        this.buffer.readBytes(buffer, length);
    }

    @Override
    public void skipBytes(int length) {
        buffer.skipBytes(length);
    }

    @Override
    public void close() {
        this.buffer.release();
    }
}
