package ru.yandex.solomon.slog.compression;

import java.util.function.Consumer;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;

/**
 * @author Vladimir Gordiychuk
 */
public class UncompressedEncodeStream implements EncodeStream {
    private final ByteBuf buffer;
    private boolean build;

    public UncompressedEncodeStream(ByteBufAllocator allocator, int bufferSize, int maxBufferSize) {
        this.buffer = allocator.buffer(bufferSize, maxBufferSize);
    }

    @Override
    public void writeHeader(Consumer<ByteBuf> writer) {
        writer.accept(buffer);
    }

    @Override
    public void writeByte(byte value) {
        flushIfNeeded(1);
        buffer.writeByte(value);
    }

    @Override
    public void writeIntLe(int value) {
        flushIfNeeded(4);
        buffer.writeIntLE(value);
    }

    @Override
    public void writeShortLe(int value) {
        flushIfNeeded(2);
        buffer.writeShortLE(value);
    }

    @Override
    public void writeLongLe(long value) {
        flushIfNeeded(8);
        buffer.writeLongLE(value);
    }

    @Override
    public void writeDouble(double value) {
        writeLongLe(Double.doubleToLongBits(value));
    }

    @Override
    public void writeString(String value) {
        flushIfNeeded(value.length() + 1);
        // TODO: check that string is a 7-bit ASCII string
        for (int i = 0; i < value.length(); i++) {
            buffer.writeByte((byte) (value.charAt(i) & 0x7f));
        }
        buffer.writeByte((byte) 0);
    }

    @Override
    public void writeVarint32(int value) {
        flushIfNeeded(5);
        while (true) {
            if ((value & ~0x7f) == 0) {
                buffer.writeByte((byte) value);
                return;
            } else {
                buffer.writeByte((byte) ((value & 0x7f) | 0x80));
                value >>>= 7;
            }
        }
    }

    @Override
    public void write(byte[] buffer, int offset, int length) {
        flushIfNeeded(1);
        while (length != 0) {
            final int remaining = this.buffer.maxWritableBytes();
            if (length <= remaining) {
                this.buffer.writeBytes(buffer, offset, length);
                return;
            }

            this.buffer.writeBytes(buffer, offset, remaining);
            offset += remaining;
            length -= remaining;

            doFlush(this.buffer);
        }
    }

    @Override
    public void write(ByteBuf buffer, int offset, int length) {
        flushIfNeeded(1);
        while (length != 0) {
            final int remaining = this.buffer.maxWritableBytes();
            if (length <= remaining) {
                this.buffer.writeBytes(buffer, offset, length);
                return;
            }

            this.buffer.writeBytes(buffer, offset, remaining);
            offset += remaining;
            length -= remaining;
            doFlush(this.buffer);
        }
    }

    @Override
    public void write(DecodeStream from, int length) {
        flushIfNeeded(1);
        while (length != 0) {
            final int remaining = this.buffer.maxWritableBytes();
            if (length <= remaining) {
                from.read(this.buffer, length);
                return;
            }

            from.read(buffer, remaining);
            length -= remaining;
            doFlush(this.buffer);
        }
    }

    @Override
    public ByteBuf finish() {
        doFlush(buffer);
        build = true;
        return buffer;
    }

    @Override
    public ByteBuf finishStream() {
        doFlush(buffer);
        build = true;
        return buffer;
    }

    private void flushIfNeeded(int i) {
        if (buffer.maxWritableBytes() < i) {
            doFlush(buffer);
        }
    }

    protected void doFlush(ByteBuf buffer) {
    }

    @Override
    public void close() {
        if (build) {
            return;
        }
        buffer.release();
    }
}
