package ru.yandex.solomon.slog.compression.alg;

import java.nio.ByteBuffer;

import io.netty.buffer.ByteBuf;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class AbstractFrameCompressor implements Compressor {
    public static final int FRAME_HEADER_SIZE = 4 + 4;
    public static final int FRAME_FOOTER_SIZE = 4;

    @Override
    public final void compress(ByteBuf src, ByteBuf dst) {
        int flushableBytes = src.readableBytes();
        if (flushableBytes == 0) {
            return;
        }

        // (1) compress
        var srcNio = src.nioBuffer(src.readerIndex(), flushableBytes);
        final int bufSize = maxCompressedLength(flushableBytes) + FRAME_HEADER_SIZE + FRAME_FOOTER_SIZE;
        dst.ensureWritable(bufSize);
        final int idx = dst.writerIndex();
        ByteBuffer dstNio = dst.nioBuffer(idx + FRAME_HEADER_SIZE, dst.writableBytes() - FRAME_HEADER_SIZE);
        int pos = dstNio.position();
        compress(srcNio, dstNio);
        int compressedLength = dstNio.position() - pos;

        // (2) add header
        dst.writeIntLE(compressedLength);
        dst.writeIntLE(flushableBytes);

        // (3) add footer
        dst.writerIndex(idx + FRAME_HEADER_SIZE + compressedLength);
        int checkSum = checksum(dstNio, pos, compressedLength);
        dst.writeIntLE(checkSum);
    }

    @Override
    public void finish(ByteBuf dst) {
        dst.writeIntLE(0);
        dst.writeIntLE(0);
        dst.writeIntLE(0);
    }

    protected abstract void compress(ByteBuffer src, ByteBuffer dst);
    protected abstract int checksum(ByteBuffer dst, int offset, int length);
    protected abstract int maxCompressedLength(int bytes);
}
