package ru.yandex.solomon.slog.compression.alg;

import java.nio.ByteBuffer;

import io.netty.buffer.ByteBuf;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class AbstractFrameDecompressor implements Decompressor {

    @Override
    public void decompress(ByteBuf src, ByteBuf dst) {
        int readableBytes = src.readableBytes();
        if (readableBytes == 0) {
            return;
        }

        int compressedLength = src.readIntLE();
        int uncompressedLength = src.readIntLE();
        if (compressedLength == 0 && uncompressedLength == 0) {
            int checksum = src.readIntLE();
            if (checksum != 0) {
                throw new RuntimeException("checksum mismatch");
            }
            return;
        }

        dst.ensureWritable(uncompressedLength);
        int expectedBytes = compressedLength + AbstractFrameCompressor.FRAME_HEADER_SIZE + AbstractFrameCompressor.FRAME_FOOTER_SIZE;
        if (readableBytes < expectedBytes) {
            throw new RuntimeException("Corrupted chunk, expected size "+ compressedLength + " but was " + readableBytes);
        }
        var srcNio = src.nioBuffer(src.readerIndex(), compressedLength);
        var checksum = checksum(srcNio, srcNio.position(), compressedLength);
        if (checksum != src.getIntLE(src.readerIndex() + compressedLength)) {
            throw new RuntimeException("checksum mismatch");
        }
        var dstNio = dst.nioBuffer(dst.writerIndex(), uncompressedLength);
        decompress(srcNio, dstNio);
        src.skipBytes(compressedLength + AbstractFrameCompressor.FRAME_FOOTER_SIZE);
        dst.writerIndex(dst.writerIndex() + uncompressedLength);
    }

    protected abstract void decompress(ByteBuffer src, ByteBuffer dst);
    protected abstract int checksum(ByteBuffer dst, int offset, int length);
}
