package ru.yandex.solomon.slog.compression.alg;

import java.nio.ByteBuffer;

import com.github.luben.zstd.Zstd;

import ru.yandex.solomon.slog.compression.checksum.XxHash32Checksum;

/**
 * @author Vladimir Gordiychuk
 * @see ru.yandex.monlib.metrics.encode.spack.compression.FrameEncodeStream
 * @see ru.yandex.monlib.metrics.encode.spack.compression.ZstdEncodeStream
 */
public class ZstdCompressor extends AbstractFrameCompressor {
    private static final int LEVEL = 11;
    private final XxHash32Checksum checksumAlg = XxHash32Checksum.INSTANCE;

    public static ZstdCompressor newInstance() {
        return new ZstdCompressor();
    }

    @Override
    protected void compress(ByteBuffer src, ByteBuffer dst) {
        long compressedSize;
        if (src.isDirect() && dst.isDirect()) {
            compressedSize = Zstd.compress(dst, src, LEVEL);
            ensureNotError(compressedSize);
        } else {
            int pos = dst.position();
            compressedSize = Zstd.compressByteArray(
                dst.array(), dst.arrayOffset(), dst.limit(),
                src.array(), src.arrayOffset(), src.remaining(),
                LEVEL);

            ensureNotError(compressedSize);
            dst.position(pos + Math.toIntExact(compressedSize));
        }
    }

    private void ensureNotError(long compressedSize) {
        if (Zstd.isError(compressedSize)) {
            throw new RuntimeException("cannot compress block with zstd: " + Zstd.getErrorName(compressedSize));
        }
    }

    @Override
    protected int checksum(ByteBuffer dst, int offset, int length) {
        return checksumAlg.calc(dst, offset, length);
    }

    @Override
    protected int maxCompressedLength(int bytes) {
        return Math.toIntExact(Zstd.compressBound(bytes));
    }
}
