package ru.yandex.solomon.slog.compression.alg;

import java.nio.ByteBuffer;

import com.github.luben.zstd.Zstd;

import ru.yandex.solomon.slog.compression.checksum.XxHash32Checksum;

/**
 * @author Vladimir Gordiychuk
 * @see ru.yandex.monlib.metrics.encode.spack.compression.FrameEncodeStream
 * @see ru.yandex.monlib.metrics.encode.spack.compression.ZstdEncodeStream
 */
public class ZstdDecompressor extends AbstractFrameDecompressor {
    private final XxHash32Checksum checksumAlg = XxHash32Checksum.INSTANCE;

    public static Decompressor newInstance() {
        return new ZstdDecompressor();
    }

    private ZstdDecompressor() {
    }

    @Override
    protected void decompress(ByteBuffer src, ByteBuffer dst) {
        long decompressedSize;
        if (src.isDirect() && dst.isDirect()) {
            decompressedSize = Zstd.decompress(dst, src);
            ensureNotError(decompressedSize);
        } else {
            int pos = dst.position();
            decompressedSize = Zstd.decompressByteArray(
                dst.array(), dst.arrayOffset(), dst.limit(),
                src.array(), src.arrayOffset(), src.remaining());

            ensureNotError(decompressedSize);
            dst.position(pos + Math.toIntExact(decompressedSize));
        }
    }

    @Override
    protected int checksum(ByteBuffer dst, int offset, int length) {
        return checksumAlg.calc(dst, offset, length);
    }

    private void ensureNotError(long compressedSize) {
        if (Zstd.isError(compressedSize)) {
            throw new RuntimeException("cannot compress block with zstd: " + Zstd.getErrorName(compressedSize));
        }
    }
}
