package ru.yandex.solomon.slog;

import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.CountColumnRandomData;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class LogDataBuilderTest {
    @Parameterized.Parameters(name = "{0}")
    public static Object[] data() {
        return Stream.of(MetricType.values())
            .filter(k -> k != MetricType.UNKNOWN)
            .toArray();
    }

    @Parameterized.Parameter
    public MetricType type;
    private TimePrecision timePrecision;
    private int numId;
    private long commonTsMillis;
    private int stepMillis;

    @Before
    public void setUp() throws Exception {
        var random = ThreadLocalRandom.current();
        timePrecision = TimePrecision.values()[random.nextInt(TimePrecision.values().length)];
        numId = random.nextInt();
        stepMillis = random.nextInt(1, 60) * 1000;
        commonTsMillis = truncateTs(System.currentTimeMillis() - random.nextInt(stepMillis));
    }

    @Test
    public void none() {
        var archive = archive();
        check(archive);
    }

    @Test
    public void one() {
        var archive = archive();
        archive.addRecord(point());
        check(archive);
    }

    @Test
    public void oneAndCount() {
        var archive = archive();
        var point = point();
        point.setCount(CountColumnRandomData.randomCount());
        archive.addRecord(point());
        check(archive);
    }

    @Test
    public void oneAndMergeCount() {
        var archive = archive();
        var point = point();
        point.setCount(CountColumnRandomData.randomCount());
        point.setMerge(true);
        check(archive);
    }

    @Test
    public void many() {
        var archive = archive();
        for (int index = 0; index < 100; index++) {
            archive.addRecord(point());
        }
        archive.sortAndMerge();
        check(archive);
    }

    @Test
    public void manyAndCount() {
        var archive = archive();
        for (int index =0; index < 100; index++) {
            var point = point();
            point.setCount(CountColumnRandomData.randomCount());
            archive.addRecord(point());
        }
        archive.sortAndMerge();
        check(archive);
    }

    @Test
    public void manyMergeAndCount() {
        var archive = archive();
        for (int index =0; index < 100; index++) {
            var point = point();
            point.setMerge(true);
            point.setCount(CountColumnRandomData.randomCount());
            archive.addRecord(point());
        }
        archive.sortAndMerge();
        check(archive);
    }

    private MetricArchiveMutable archive() {
        MetricArchiveMutable archive = new MetricArchiveMutable();
        archive.setType(MetricTypeConverter.toNotNullProto(type));
        archive.setOwnerShardId(numId);
        return archive;
    }

    private AggrPoint point() {
        int mask = StockpileColumns.minColumnSet(MetricTypeConverter.toNotNullProto(type));
        var point = randomPoint(mask);
        point.tsMillis = truncateTs(ThreadLocalRandom.current().nextBoolean() ? commonTsMillis : point.tsMillis);
        point.setStepMillis(stepMillis);
        point.valueDenom = ValueColumn.DEFAULT_DENOM;
        return point;
    }

    private long truncateTs(long tsMillis) {
        if (timePrecision == TimePrecision.MILLIS) {
            return tsMillis;
        }
        return TimeUnit.SECONDS.toMillis(TimeUnit.MILLISECONDS.toSeconds(tsMillis));
    }

    private void check(MetricArchiveMutable archive) {
        checkUnpacked(archive);
    }

    private void checkUnpacked(MetricArchiveMutable archive) {
        ByteBuf buffer = null;
        try (var builder = makeUnpackedBuilder()) {
            int flags = logFlags(archive);
            var type = MetricTypeConverter.fromProto(archive.getType());
            int size = builder.onTimeSeries(type, flags, AggrGraphDataArrayList.of(archive));
            buffer = builder.build();

            var header = new LogDataHeader(buffer);
            {
                assertEquals(numId, header.numId);
                assertEquals(timePrecision, header.timePrecision);
                assertEquals(commonTsMillis, header.commonTsMillis);
                assertEquals(stepMillis, header.stepMillis);
                assertEquals(CompressionAlg.NONE, header.compressionAlg);
                assertEquals(1, header.metricsCount);
                assertEquals(archive.getRecordCount(), header.pointsCount);
            }
            assertEquals(size, buffer.readableBytes());
            assertData(header, buffer, archive);
        } finally {
            release(buffer);
        }
    }

    private LogDataBuilder makeUnpackedBuilder() {
        return new LogDataBuilderImpl(CompressionAlg.NONE, ByteBufAllocator.DEFAULT, timePrecision, numId, commonTsMillis, stepMillis);
    }

    private int logFlags(MetricArchiveMutable archive) {
        int flags = 0;
        if (archive.hasColumn(StockpileColumn.MERGE)) {
            flags |= LogFlags.MERGE.mask;
        }
        if (archive.hasColumn(StockpileColumn.COUNT)) {
            flags |= LogFlags.COUNT.mask;
        }
        return flags;
    }

    private void assertData(LogDataHeader header, ByteBuf buffer, MetricArchiveMutable expected) {
        try (var it = new LogDataIteratorImpl(header, buffer)) {
            assertTrue(it.hasNext());
            var actual = new MetricArchiveMutable();
            it.next(actual);
            assertEquals(expected, actual);
            assertFalse(it.hasNext());
        }
    }

    private void release(@Nullable ByteBuf buffer) {
        if (buffer != null) {
            buffer.release();
        }
    }
}
