package ru.yandex.solomon.slog;

import java.util.concurrent.ThreadLocalRandom;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.Test;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class LogDataHeaderTest {
    @Test
    public void serializeDeserialize() {
        for (int index = 0; index < 10; index++) {
            LogDataHeader source = randomHeader();

            var one = serialize(source);
            var two = serialize(source);
            assertEquals(one, two);

            var result = new LogDataHeader(one);
            assertEquals(0, one.readableBytes());
            assertEquals(result.numId, source.numId);
            assertEquals(result.version, source.version);
            assertEquals(result.commonTsMillis, source.commonTsMillis);
            assertEquals(result.stepMillis, source.stepMillis);
            assertEquals(result.timePrecision, source.timePrecision);
            assertEquals(result.compressionAlg, source.compressionAlg);
            assertEquals(result.dataCodingScheme, source.dataCodingScheme);
            assertEquals(result.metricsCount, source.metricsCount);
            assertEquals(result.pointsCount, source.pointsCount);
        }
    }

    @Test
    public void changeMetricsCount() {
        var source = randomHeader();
        var serialized = serialize(source);
        source.setMetricsCount(serialized.readerIndex(), serialized, 42);

        var result = new LogDataHeader(serialized);
        assertEquals(0, serialized.readableBytes());
        assertEquals(source.numId, result.numId);
        assertEquals(source.version, result.version);
        assertEquals(source.commonTsMillis, result.commonTsMillis);
        assertEquals(source.stepMillis, result.stepMillis);
        assertEquals(source.timePrecision, result.timePrecision);
        assertEquals(source.compressionAlg, result.compressionAlg);
        assertEquals(source.dataCodingScheme, result.dataCodingScheme);
        assertEquals(42, result.metricsCount);
        assertEquals(source.pointsCount, result.pointsCount);
    }

    @Test
    public void changePointsCount() {
        var source = randomHeader();
        var serialized = serialize(source);
        source.setPointsCount(serialized.readerIndex(), serialized, 42);

        var result = new LogDataHeader(serialized);
        assertEquals(0, serialized.readableBytes());
        assertEquals(source.numId, result.numId);
        assertEquals(source.version, result.version);
        assertEquals(source.commonTsMillis, result.commonTsMillis);
        assertEquals(source.stepMillis, result.stepMillis);
        assertEquals(source.timePrecision, result.timePrecision);
        assertEquals(source.compressionAlg, result.compressionAlg);
        assertEquals(source.dataCodingScheme, result.dataCodingScheme);
        assertEquals(source.metricsCount, result.metricsCount);
        assertEquals(42, result.pointsCount);
    }

    @Test(expected = CorruptedBinaryDataRuntimeException.class)
    public void deserializeEmpty() {
        new LogDataHeader(UnpooledByteBufAllocator.DEFAULT.buffer());
    }

    @Test(expected = CorruptedBinaryDataRuntimeException.class)
    public void deserializeSmall() {
        new LogDataHeader(UnpooledByteBufAllocator.DEFAULT.buffer().writeBytes("junk".getBytes()));
    }

    @Test(expected = CorruptedBinaryDataRuntimeException.class)
    public void deserializeHugeJunk() {
        new LogDataHeader(UnpooledByteBufAllocator.DEFAULT.buffer()
            .writeBytes("huge amount of junk that will be instead of original header".getBytes())
            .writeZero(128 << 10));
    }

    private ByteBuf serialize(LogDataHeader source) {
        var size = source.size();
        var result = UnpooledByteBufAllocator.DEFAULT.buffer(size, size);
        source.writeTo(result);
        assertEquals(size, result.readableBytes());
        return result;
    }

    private LogDataHeader randomHeader() {
        var random = ThreadLocalRandom.current();
        return new LogDataHeader(
            random.nextInt(),
            random.nextLong(),
            random.nextInt(0, Integer.MAX_VALUE),
            TimePrecision.values()[random.nextInt(TimePrecision.values().length)],
            CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)],
            DataCodingScheme.values()[random.nextInt(DataCodingScheme.values().length)],
            random.nextInt(0, Integer.MAX_VALUE),
            random.nextInt(0, Integer.MAX_VALUE));
    }
}
