package ru.yandex.solomon.slog;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import io.netty.buffer.ByteBufAllocator;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.CountColumnRandomData;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.point.column.TsRandomData;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMetricType;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class LogDataIteratorImplTest {

    private TimePrecision timePrecision;
    private int numId;
    private long commonTsMillis;
    private int stepMillis;
    private LogDataBuilder builder;

    @Before
    public void setUp() throws Exception {
        var random = ThreadLocalRandom.current();
        var alg = CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)];
        timePrecision = TimePrecision.values()[random.nextInt(TimePrecision.values().length)];
        numId = random.nextInt();
        commonTsMillis = System.currentTimeMillis() - random.nextInt(100);
        stepMillis = random.nextInt(1, 60) * 1000;
        builder = new LogDataBuilderImpl(alg, ByteBufAllocator.DEFAULT, timePrecision, numId, commonTsMillis, stepMillis);
    }

    @After
    public void tearDown() throws Exception {
        builder.close();
    }

    @Test
    public void readOneNoTs() {
        var expected = archive();
        expected.setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);
        expected.addRecord(AggrPoint.builder()
            .time(commonTsMillis)
            .doubleValue(42.05)
            .stepMillis(stepMillis)
            .build());

        var point = new AggrPoint();
        point.valueNum = 42.05;
        builder.onPoint(MetricType.DGAUGE, 0, point);
        var parsed = parse();
        assertEquals(List.of(expected), parsed);
    }

    @Test
    public void readOneWithTs() {
        long ts0 = truncateTs(System.currentTimeMillis());
        var expected = archive();
        expected.setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);
        expected.addRecord(AggrPoint.builder()
            .time(ts0)
            .doubleValue(43)
            .stepMillis(stepMillis)
            .build());

        var point = new AggrPoint();
        point.tsMillis = ts0;
        point.valueNum = 43;
        builder.onPoint(MetricType.DGAUGE, 0, point);
        var parsed = parse();
        assertEquals(List.of(expected), parsed);
    }

    @Test
    public void randomByOnePoint() {
        var expected = new ArrayList<MetricArchiveMutable>();
        for (int index = 0; index < 1000; index++) {
            var type = randomMetricType();
            var mask = StockpileColumns.minColumnSet(type);
            var point = randomPoint(mask);
            point.setStepMillis(stepMillis);
            point.valueDenom = ValueColumn.DEFAULT_DENOM;
            point.tsMillis = truncateTs(point.tsMillis);
            builder.onPoint(MetricTypeConverter.fromProto(type), 0, point);

            var archive = archive();
            archive.addRecord(point);
            expected.add(archive);
        }
        var parsed = parse();
        assertArrayEquals(expected.toArray(), parsed.toArray());
    }

    @Test
    public void randomByMultiplePoint() {
        var random = ThreadLocalRandom.current();
        var expected = new ArrayList<MetricArchiveMutable>();
        var point = RecyclableAggrPoint.newInstance();
        for (int metricIndex = 0; metricIndex < 1000; metricIndex++) {
            var type = randomMetricType();
            var mask = StockpileColumns.minColumnSet(type);
            var ts0 = truncateTs(TsRandomData.randomTs(random));
            var archive = archive();
            for (int pointIndex = 0; pointIndex < random.nextInt(1, 100); pointIndex++) {
                randomPoint(point, mask);
                point.valueDenom = ValueColumn.DEFAULT_DENOM;
                point.setTsMillis(ts0 + (stepMillis * pointIndex));
                point.setStepMillis(stepMillis);
                archive.addRecord(point);
            }
            builder.onTimeSeries(MetricTypeConverter.fromProto(type), 0, AggrGraphDataArrayList.of(archive));
            expected.add(archive);
        }
        point.recycle();
        var parsed = parse();
        assertArrayEquals(expected.toArray(), parsed.toArray());
    }

    @Test
    public void readOnePlusDenom() {
        long ts0 = truncateTs(System.currentTimeMillis());
        var expected = archive();
        expected.setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);
        expected.addRecord(AggrPoint.builder()
            .time(ts0)
            .doubleValue(43, 15_000)
            .stepMillis(stepMillis)
            .build());

        var point = new AggrPoint();
        point.tsMillis = ts0;
        point.valueNum = 43;
        point.valueDenom = 15_000;
        builder.onPoint(MetricType.DGAUGE, LogFlags.DENOM.mask, point);

        var parsed = parse();
        assertEquals(List.of(expected), parsed);
    }

    @Test
    public void readOnePlusDenomPlusMerge() {
        long ts0 = truncateTs(System.currentTimeMillis());
        var expected = archive();
        expected.setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);
        expected.addRecord(AggrPoint.builder()
            .time(ts0)
            .doubleValue(50, 15_000)
            .stepMillis(stepMillis)
            .merge(true)
            .count(145)
            .build());

        var point = new AggrPoint();
        point.tsMillis = ts0;
        point.valueNum = 50;
        point.valueDenom = 15_000;
        point.count = 145;
        point.merge = true;
        builder.onPoint(MetricType.DGAUGE, LogFlags.DENOM.mask | LogFlags.MERGE.mask | LogFlags.COUNT.mask, point);

        var parsed = parse();
        assertEquals(List.of(expected), parsed);
    }

    @Test
    public void readOnePlusAggregate() {
        var expected = new ArrayList<MetricArchiveMutable>();
        for (int index = 0; index < 1000; index++) {
            var type = randomMetricType();
            var mask = StockpileColumns.minColumnSet(type);
            var point = randomPoint(mask);
            point.setMerge(true);
            point.setCount(CountColumnRandomData.randomCount(ThreadLocalRandom.current()));
            point.setStepMillis(stepMillis);
            point.valueDenom = ValueColumn.DEFAULT_DENOM;
            point.tsMillis = truncateTs(point.tsMillis);
            builder.onPoint(MetricTypeConverter.fromProto(type), LogFlags.MERGE.mask | LogFlags.COUNT.mask, point);

            var archive = archive();
            archive.addRecord(point);
            expected.add(archive);
        }
        var parsed = parse();
        assertArrayEquals(expected.toArray(), parsed.toArray());
    }

    @Test
    public void readManyPlusAggregate() {
        var random = ThreadLocalRandom.current();
        var expected = new ArrayList<MetricArchiveMutable>();
        var point = RecyclableAggrPoint.newInstance();
        for (int metricIndex = 0; metricIndex < 100; metricIndex++) {
            var type = randomMetricType();
            var mask = StockpileColumns.minColumnSet(type);
            var ts0 = truncateTs(TsRandomData.randomTs(random));
            var archive = archive();
            var pointCnt = random.nextInt(1, 100);
            var list = new AggrGraphDataArrayList(mask, pointCnt);
            for (int pointIndex = 0; pointIndex < pointCnt; pointIndex++) {
                randomPoint(point, mask);
                point.valueDenom = ValueColumn.DEFAULT_DENOM;
                point.setTsMillis(ts0 + (stepMillis * pointIndex));
                point.setStepMillis(stepMillis);
                point.setMerge(true);
                point.setCount(CountColumnRandomData.randomCount(random));
                archive.addRecord(point);
                list.addRecord(point);
            }
            builder.onTimeSeries(MetricTypeConverter.fromProto(type), LogFlags.MERGE.mask | LogFlags.COUNT.mask, list);
            expected.add(archive);
            list.clear();
        }
        point.recycle();
        var parsed = parse();
        assertArrayEquals(expected.toArray(), parsed.toArray());
    }

    @Test
    public void readManyPlusDenom() {
        long ts0 = truncateTs(System.currentTimeMillis());
        var expected = archive();
        expected.setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);
        expected.addRecord(AggrPoint.builder()
            .time(ts0)
            .doubleValue(43, 15_000)
            .stepMillis(stepMillis)
            .build());
        expected.addRecord(AggrPoint.builder()
            .time(ts0 + stepMillis)
            .doubleValue(50, 15_000)
            .stepMillis(stepMillis)
            .build());

        AggrGraphDataArrayList timeseries = AggrGraphDataArrayList.of(expected);
        builder.onTimeSeries(MetricType.DGAUGE, LogFlags.DENOM.mask, timeseries);

        var parsed = parse();
        assertArrayEquals(new Object[]{expected}, parsed.toArray());
    }

    private long truncateTs(long tsMillis) {
        if (timePrecision == TimePrecision.MILLIS) {
            return tsMillis;
        }
        return TimeUnit.SECONDS.toMillis(TimeUnit.MILLISECONDS.toSeconds(tsMillis));
    }

    private MetricArchiveMutable archive() {
        var archive = new MetricArchiveMutable();
        archive.setOwnerShardId(numId);
        return archive;
    }

    private List<MetricArchiveMutable> parse() {
        var buffer = builder.build();
        try (var it = LogDataIterator.create(buffer)) {
            List<MetricArchiveMutable> result = new ArrayList<>();
            while (it.hasNext()) {
                MetricArchiveMutable archive = new MetricArchiveMutable();
                it.next(archive);
                result.add(archive);
            }
            return result;
        } finally {
            buffer.release();
        }
    }
}
