package ru.yandex.solomon.slog;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StepColumn;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.stockpile.api.EDecimPolicy;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMetricType;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;


/**
 * @author Vladimir Gordiychuk
 */
public class LogsTest {

    private int numId;
    private CompressionAlg compression;
    private int producerId;
    private long producerSeqNo;
    private long commonTs;
    private EDecimPolicy decimPolicy;
    private int stepMillis = 10_000;

    @Before
    public void setUp() {
        var random = ThreadLocalRandom.current();
        numId = random.nextInt();
        compression = CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)];
        commonTs = System.currentTimeMillis();
        producerId = ThreadLocalRandom.current().nextInt();
        producerSeqNo = ThreadLocalRandom.current().nextLong();
        decimPolicy = EDecimPolicy.values()[ThreadLocalRandom.current().nextInt(EDecimPolicy.values().length - 1)];
    }

    @Test
    public void combineMeta() {
        var expectedLeft = randomMeta(100);
        var expectedRight = randomMeta(42);
        var expected = Lists.newArrayList(Iterables.concat(expectedLeft, expectedRight));

        ByteBuf combined = Logs.combineResolvedMeta(encodeMeta(expectedLeft), encodeMeta(expectedRight));
        assertMeta(expected, combined);
    }

    @Test
    public void combineData() {
        var expectedLeft = randomData(100);
        var expectedRight = randomData(42);
        var expected = Lists.newArrayList(Iterables.concat(expectedLeft, expectedRight));

        ByteBuf combined = Logs.combineData(encodeData(expectedLeft), encodeData(expectedRight));
        assertData(expected, combined);
    }

    @Test
    public void combineSnapshotData() {
        var expectedLeft = randomData(100);
        var expectedRight = randomData(42);
        var expected = Lists.newArrayList(Iterables.concat(expectedLeft, expectedRight));

        ByteBuf combined = Logs.combineSnapshotData(encodeSnapshotData(expectedLeft), encodeSnapshotData(expectedRight));
        assertData(expected, combined);
    }

    @Test
    public void combineLog() {
        var leftMeta = randomMeta(42);
        var leftData = randomData(42);
        var rightMeta = randomMeta(100);
        var rightData = randomData(100);

        var expectedMeta = Lists.newArrayList(Iterables.concat(leftMeta, rightMeta));
        var expectedData = Lists.newArrayList(Iterables.concat(leftData, rightData));

        var log = Logs.combineResolvedLog(encode(leftMeta, leftData), encode(rightMeta, rightData));
        assertMeta(expectedMeta, log.meta);
        assertData(expectedData, log.data);
    }

    @Test
    public void combineSnapshotLog() {
        var leftMeta = randomMeta(42);
        var leftData = randomData(42);
        var rightMeta = randomMeta(100);
        var rightData = randomData(100);

        var expectedMeta = Lists.newArrayList(Iterables.concat(leftMeta, rightMeta));
        var expectedData = Lists.newArrayList(Iterables.concat(leftData, rightData));

        var log = Logs.combineResolvedLog(encodeSnapshot(leftMeta, leftData), encodeSnapshot(rightMeta, rightData));
        assertMeta(expectedMeta, log.meta);
        assertData(expectedData, log.data);
    }

    private void assertMeta(ArrayList<ResolvedLogMetaRecord> expected, ByteBuf combined) {
        try {
            var header = new ResolvedLogMetaHeader(combined);
            assertEquals(numId, header.numId);
            assertEquals(compression, header.compressionAlg);
            assertEquals(producerId, header.producerId);
            assertEquals(producerSeqNo, header.producerSeqNo);
            assertEquals(decimPolicy, header.decimPolicy);
            assertEquals(expected.size(), header.metricsCount);
            assertEquals(expected.stream().mapToInt(value -> value.points).sum(), header.pointsCount);
            var record = new ResolvedLogMetaRecord();
            var expectIt = expected.iterator();
            try (var it = new ResolvedLogMetaIteratorImpl(header, combined)) {
                while (it.next(record)) {
                    assertTrue(expectIt.hasNext());
                    assertEquals(expectIt.next(), record);
                }
                assertFalse(expectIt.hasNext());
            }
        } finally {
            combined.release();
        }
    }

    private void assertData(ArrayList<MetricArchiveMutable> expected, ByteBuf combined) {
        try {
            var expectedIt = expected.iterator();
            try (var it = LogDataIterator.create(combined)) {
                assertEquals(numId, it.getNumId());
                assertEquals(expected.size(), it.getMetricsCount());
                assertEquals(expected.stream().mapToInt(MetricArchiveMutable::getRecordCount).sum(), it.getPointsCount());

                while (it.hasNext()) {
                    assertTrue(expectedIt.hasNext());
                    var expectedArchive = expectedIt.next();
                    var archive = new MetricArchiveMutable();
                    it.next(archive);
                    assertEquals(expectedArchive.getType(), archive.getType());
                    assertEquals(expectedArchive.getOwnerShardId(), archive.getOwnerShardId());
                    assertEquals(expectedArchive, archive);
                }
                assertFalse(expectedIt.hasNext());
            }
        } finally {
            combined.release();
        }
    }

    private Log encode(List<ResolvedLogMetaRecord> meta, List<MetricArchiveMutable> data) {
        return new Log(numId, encodeMeta(meta), encodeData(data));
    }

    private Log encodeSnapshot(List<ResolvedLogMetaRecord> meta, List<MetricArchiveMutable> data) {
        return new Log(numId, encodeMeta(meta), encodeSnapshotData(data));
    }

    private ByteBuf encodeMeta(List<ResolvedLogMetaRecord> records) {
        var header = new ResolvedLogMetaHeader(numId, compression)
                .setProducerId(producerId)
                .setProducerSeqNo(producerSeqNo)
                .setDecimPolicy(decimPolicy);

        try (var builder = new ResolvedLogMetaBuilderImpl(header, ByteBufAllocator.DEFAULT)) {
            for (var record : records) {
                builder.onMetric(record.type, record.localId, record.points, record.dataSize);
            }
            return builder.build();
        }
    }

    private ByteBuf encodeData(List<MetricArchiveMutable> archives) {
        try (var builder = new LogDataBuilderImpl(compression, ByteBufAllocator.DEFAULT, TimePrecision.MILLIS, numId, commonTs, stepMillis)) {
            for (var archive : archives) {
                var type = MetricTypeConverter.fromProto(archive.getType());
                builder.onTimeSeries(type, 0, AggrGraphDataArrayList.of(archive));
            }
            return builder.build();
        }
    }

    private ByteBuf encodeSnapshotData(List<MetricArchiveMutable> archives) {
        try (var builder = new SnapshotLogDataBuilderImpl(compression, numId, ByteBufAllocator.DEFAULT)) {
            for (var archive : archives) {
                builder.onTimeSeries(archive.toImmutableNoCopy());
            }
            return builder.build();
        }
    }

    private List<ResolvedLogMetaRecord> randomMeta(int count) {
        return IntStream.range(0, count)
                .mapToObj(ignore -> randomMeta())
                .collect(Collectors.toList());
    }

    private ResolvedLogMetaRecord randomMeta() {
        var random = ThreadLocalRandom.current();
        var record = new ResolvedLogMetaRecord();
        record.type = MetricType.values()[random.nextInt(1, MetricType.values().length)];
        record.dataSize = random.nextInt(1, 256);
        record.localId = random.nextLong();
        record.points = random.nextInt(1, 100);
        return record;
    }

    private List<MetricArchiveMutable> randomData(int count) {
        return IntStream.range(0, count)
                .mapToObj(ignore -> randomData())
                .collect(Collectors.toList());
    }

    private MetricArchiveMutable randomData() {
        var type = randomMetricType();
        int mask = StockpileColumns.minColumnSet(type) | StepColumn.mask;
        int points = ThreadLocalRandom.current().nextInt(1, 100);
        MetricArchiveMutable archive = new MetricArchiveMutable();
        archive.setOwnerShardId(numId);
        archive.setType(type);
        archive.ensureCapacity(mask, points);
        long tsMillis = commonTs;
        var tmp = new AggrPoint();
        for (int index = 0; index < points; index++) {
            randomPoint(tmp, mask);
            tmp.valueDenom = ValueColumn.DEFAULT_DENOM;
            tmp.tsMillis = tsMillis;
            tmp.stepMillis = stepMillis;
            archive.addRecordData(mask, tmp);
            tsMillis += stepMillis;
        }
        return archive;
    }
}
