package ru.yandex.solomon.slog;

import java.util.concurrent.ThreadLocalRandom;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.Test;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;
import ru.yandex.stockpile.api.EDecimPolicy;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class ResolvedLogMetaHeaderTest {
    @Test
    public void serializeDeserialize() {
        for (int index = 0; index < 10; index++) {
            var random = ThreadLocalRandom.current();
            var source = new ResolvedLogMetaHeader(random.nextInt(), CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)]);
            source.pointsCount = random.nextInt(0, Integer.MAX_VALUE);
            source.metricsCount = random.nextInt(0, Integer.MAX_VALUE);
            source.producerId = random.nextInt(4096);
            source.producerSeqNo = random.nextLong();
            source.decimPolicy = EDecimPolicy.values()[random.nextInt(EDecimPolicy.values().length - 1)];

            var one = serialize(source);
            var two = serialize(source);
            assertEquals(one, two);

            var result = new ResolvedLogMetaHeader(one);
            assertEquals(0, one.readableBytes());
            assertEquals(result.numId, source.numId);
            assertEquals(result.version, source.version);
            assertEquals(result.compressionAlg, source.compressionAlg);
            assertEquals(result.metricsCount, source.metricsCount);
            assertEquals(result.pointsCount, source.pointsCount);
            assertEquals(result.producerId, source.producerId);
            assertEquals(result.producerSeqNo, source.producerSeqNo);
            assertEquals(result.decimPolicy, source.decimPolicy);
        }
    }

    @Test(expected = CorruptedBinaryDataRuntimeException.class)
    public void deserializeEmpty() {
        new ResolvedLogMetaHeader(UnpooledByteBufAllocator.DEFAULT.buffer());
    }

    @Test(expected = CorruptedBinaryDataRuntimeException.class)
    public void deserializeSmall() {
        new ResolvedLogMetaHeader(UnpooledByteBufAllocator.DEFAULT.buffer().writeBytes("junk".getBytes()));
    }

    @Test(expected = CorruptedBinaryDataRuntimeException.class)
    public void deserializeHugeJunk() {
        new ResolvedLogMetaHeader(UnpooledByteBufAllocator.DEFAULT.buffer()
            .writeBytes("huge amount of junk that will be instead of original header".getBytes())
            .writeZero(128 << 10));
    }

    @Test
    public void patchMetricsCount() {
        var source = new ResolvedLogMetaHeader(2, CompressionAlg.LZ4);
        source.producerId = 55;
        source.producerSeqNo = 123;
        source.metricsCount = 42;
        source.pointsCount = 10;
        var buffer = serialize(source);
        try {
            source.setMetricsCount(buffer.readerIndex(), buffer, 25);
            var result = new ResolvedLogMetaHeader(buffer);
            assertEquals(0, buffer.readableBytes());
            assertEquals(source.numId, result.numId);
            assertEquals(source.version, result.version);
            assertEquals(source.compressionAlg, result.compressionAlg);
            assertEquals(source.producerId, result.producerId);
            assertEquals(source.producerSeqNo, result.producerSeqNo);
            assertEquals(25, result.metricsCount);
            assertEquals(source.pointsCount, result.pointsCount);
        } finally {
            buffer.release();
        }
    }

    @Test
    public void patchPointsCount() {
        var source = new ResolvedLogMetaHeader(2, CompressionAlg.LZ4);
        source.metricsCount = 42;
        source.pointsCount = 10;
        source.producerId = 55;
        source.producerSeqNo = 123;
        var buffer = serialize(source);
        try {
            source.setPointsCount(buffer.readerIndex(), buffer, 123);
            var result = new ResolvedLogMetaHeader(buffer);
            assertEquals(0, buffer.readableBytes());
            assertEquals(source.numId, result.numId);
            assertEquals(source.version, result.version);
            assertEquals(source.compressionAlg, result.compressionAlg);
            assertEquals(source.producerId, result.producerId);
            assertEquals(source.producerSeqNo, result.producerSeqNo);
            assertEquals(source.metricsCount, result.metricsCount);
            assertEquals(123, result.pointsCount);
        } finally {
            buffer.release();
        }
    }

    @Test
    public void readProducerIdSeqNo() {
        var random = ThreadLocalRandom.current();
        var source = new ResolvedLogMetaHeader(random.nextInt(), CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)]);
        source.pointsCount = random.nextInt(0, Integer.MAX_VALUE);
        source.metricsCount = random.nextInt(0, Integer.MAX_VALUE);
        source.producerId = random.nextInt(4096);
        source.producerSeqNo = random.nextLong();
        source.decimPolicy = EDecimPolicy.values()[random.nextInt(EDecimPolicy.values().length - 1)];

        var buffer = serialize(source);
        try {
            assertEquals(source.producerId, ResolvedLogMetaHeader.producerId(buffer));
            assertEquals(source.producerSeqNo, ResolvedLogMetaHeader.producerSeqNo(buffer));
        } finally {
            buffer.release();
        }
    }

    private ByteBuf serialize(ResolvedLogMetaHeader source) {
        int size = source.size();
        var result = UnpooledByteBufAllocator.DEFAULT.buffer(size, size);
        source.writeTo(result);
        assertEquals(size, result.readableBytes());
        return result;
    }
}
