package ru.yandex.solomon.slog;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static ru.yandex.monlib.metrics.MetricType.DGAUGE;
import static ru.yandex.monlib.metrics.MetricType.IGAUGE;

/**
 * @author Vladimir Gordiychuk
 */
public class ResolvedLogMetaIteratorImplTest {

    private ResolvedLogMetaBuilder encoder;

    @Before
    public void setUp() {
        var random = ThreadLocalRandom.current();
        var alg = CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)];
        var numId = random.nextInt();
        var header = new ResolvedLogMetaHeader(numId, alg);
        encoder = new ResolvedLogMetaBuilderImpl(header, UnpooledByteBufAllocator.DEFAULT);
    }

    @After
    public void tearDown() {
        if (encoder != null) {
            encoder.close();
        }
    }

    @Test
    public void parseOneMetric() {
        var expect = record(DGAUGE, ThreadLocalRandom.current().nextLong(), 1, 14);
        encode(expect);
        var parsed = parse(1, 1);

        assertEquals(1, parsed.size());
        Assert.assertEquals(expect, parsed.get(0));
    }

    @Test
    public void parseFewMetrics() {
        var one = record(DGAUGE, ThreadLocalRandom.current().nextLong(), 1, 13);
        var two = record(DGAUGE, ThreadLocalRandom.current().nextLong(), 2, 9);

        encode(one, two);
        var parsed = parse(2, 3);
        assertEquals(2, parsed.size());
        Assert.assertEquals(one, parsed.get(0));
        Assert.assertEquals(two, parsed.get(1));
    }

    @Test
    public void parseDifferentType() {
        var one = record(DGAUGE, ThreadLocalRandom.current().nextLong(), 1, 13);
        var two = record(IGAUGE, ThreadLocalRandom.current().nextLong(), 1, 9);

        encode(one, two);
        var parsed = parse(2, 2);
        assertEquals(2, parsed.size());
        Assert.assertEquals(one, parsed.get(0));
        Assert.assertEquals(two, parsed.get(1));
    }

    @Test
    public void parseLotsOfMetrics() {
        var random = ThreadLocalRandom.current();
        var records = IntStream.range(0, 50_000)
            .mapToObj(index -> record(DGAUGE, random.nextLong(), random.nextInt(index + 1), random.nextInt(index + 1)))
            .toArray(ResolvedLogMetaRecord[]::new);

        encode(records);

        int points = Stream.of(records).mapToInt(value -> value.points).sum();
        var parsed = parse(50_000, points);

        assertEquals(50_000, parsed.size());
        for (int index = 0; index < records.length; index++) {
            var expected = records[index];
            var actual = parsed.get(index);
            assertEquals(expected, actual);
        }
    }

    private void encode(ResolvedLogMetaRecord... records) {
        for (var record : records) {
            encoder.onMetric(record.type, record.localId, record.points, record.dataSize);
        }
    }

    private static ResolvedLogMetaRecord record(MetricType type, long localId, int points, int dataSize) {
        return new ResolvedLogMetaRecord(type, localId, points, dataSize);
    }

    private List<ResolvedLogMetaRecord> parse(int expectedMetrics, int expectedPoints) {
        ByteBuf buffer = encoder.build();
        try {
            var header = new ResolvedLogMetaHeader(buffer);
            assertEquals(expectedMetrics, header.metricsCount);
            assertEquals(expectedPoints, header.pointsCount);
            try (var it = new ResolvedLogMetaIteratorImpl(header, buffer)) {
                List<ResolvedLogMetaRecord> result = new ArrayList<>(header.metricsCount);
                var record = new ResolvedLogMetaRecord();
                while (it.next(record)) {
                    result.add(record);
                    record = new ResolvedLogMetaRecord();
                }
                assertFalse(it.next(record));
                return result;
            }
        } finally {
            encoder = null;
            buffer.release();
        }
    }
}
