package ru.yandex.solomon.slog;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.Labels;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static ru.yandex.monlib.metrics.MetricType.DGAUGE;
import static ru.yandex.monlib.metrics.MetricType.IGAUGE;

/**
 * @author Vladimir Gordiychuk
 */
public class ResolvedLogMetricsIteratorImplTest {

    private ResolvedLogMetricsBuilder encoder;

    @Before
    public void setUp() {
        var alg = CompressionAlg.values()[ThreadLocalRandom.current().nextInt(CompressionAlg.values().length)];
        var numId = ThreadLocalRandom.current().nextInt();
        encoder = new ResolvedLogMetricsBuilderImpl(numId, alg, UnpooledByteBufAllocator.DEFAULT);
    }

    @After
    public void tearDown() {
        encoder.close();
    }

    @Test
    public void parseOneMetric() {
        var expect = record(DGAUGE, Labels.of("name", "alice"), randomShardId(), randomLocalId());
        encode(expect);
        var parsed = parse(1);

        assertEquals(1, parsed.size());
        Assert.assertEquals(expect, parsed.get(0));
    }

    @Test
    public void parseFewMetrics() {
        var one = record(DGAUGE, Labels.of("name", "alice"), 1, randomLocalId());
        var two = record(DGAUGE, Labels.of("name", "bob"), 2, randomLocalId());

        encode(one, two);
        var parsed = parse(2);
        assertEquals(2, parsed.size());
        Assert.assertEquals(one, parsed.get(0));
        Assert.assertEquals(two, parsed.get(1));
    }

    @Test
    public void parseDifferentType() {
        var one = record(DGAUGE, Labels.of("name", "alice"), 1, randomLocalId());
        var two = record(IGAUGE, Labels.of("name", "bob"), 1, randomLocalId());

        encode(one, two);
        var parsed = parse(2);
        assertEquals(2, parsed.size());
        Assert.assertEquals(one, parsed.get(0));
        Assert.assertEquals(two, parsed.get(1));
    }

    private long randomLocalId() {
        return ThreadLocalRandom.current().nextLong();
    }

    private int randomShardId() {
        return ThreadLocalRandom.current().nextInt(1, 4097);
    }

    private void encode(ResolvedLogMetricsRecord... records) {
        for (var record : records) {
            encoder.onMetric(record.type, record.labels, record.shardId, record.localId);
        }
    }

    private static ResolvedLogMetricsRecord record(MetricType type, Labels labels, int shardId, long localId) {
        return new ResolvedLogMetricsRecord(type, labels, shardId, localId);
    }

    private List<ResolvedLogMetricsRecord> parse(int expectedMetrics) {
        ByteBuf buffer = encoder.build();
        try {
            var header = new ResolvedLogMetricsHeader(buffer);
            assertEquals(expectedMetrics, header.metricsCount);
            try (var it = new ResolvedLogMetricsIteratorImpl(header, buffer, Labels.allocator)) {
                List<ResolvedLogMetricsRecord> result = new ArrayList<>(header.metricsCount);
                var record = new ResolvedLogMetricsRecord();
                while (it.next(record)) {
                    result.add(record);
                    record = new ResolvedLogMetricsRecord();
                }
                assertFalse(it.next(record));
                return result;
            }
        } finally {
            buffer.release();
        }
    }
}
