package ru.yandex.solomon.slog;

import java.util.concurrent.ThreadLocalRandom;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.Test;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class SnapshotLogDataHeaderTest {
    @Test
    public void serializeDeserialize() {
        for (int index = 0; index < 10; index++) {
            SnapshotLogDataHeader source = randomHeader();

            var one = serialize(source);
            var two = serialize(source);
            assertEquals(one, two);

            var result = new SnapshotLogDataHeader(one);
            assertEquals(0, one.readableBytes());
            assertEquals(result.numId, source.numId);
            assertEquals(result.version, source.version);
            assertEquals(result.compressionAlg, source.compressionAlg);
            assertEquals(result.metricsCount, source.metricsCount);
            assertEquals(result.pointsCount, source.pointsCount);
        }
    }

    @Test
    public void changeMetricsCount() {
        var source = randomHeader();
        var serialized = serialize(source);
        source.setMetricsCount(serialized.readerIndex(), serialized, 42);

        var result = new SnapshotLogDataHeader(serialized);
        assertEquals(0, serialized.readableBytes());
        assertEquals(source.numId, result.numId);
        assertEquals(source.version, result.version);
        assertEquals(source.compressionAlg, result.compressionAlg);
        assertEquals(42, result.metricsCount);
        assertEquals(source.pointsCount, result.pointsCount);
    }

    @Test
    public void changePointsCount() {
        var source = randomHeader();
        var serialized = serialize(source);
        source.setPointsCount(serialized.readerIndex(), serialized, 42);

        var result = new SnapshotLogDataHeader(serialized);
        assertEquals(0, serialized.readableBytes());
        assertEquals(source.numId, result.numId);
        assertEquals(source.version, result.version);
        assertEquals(source.compressionAlg, result.compressionAlg);
        assertEquals(source.metricsCount, result.metricsCount);
        assertEquals(42, result.pointsCount);
    }

    @Test(expected = CorruptedBinaryDataRuntimeException.class)
    public void deserializeEmpty() {
        new SnapshotLogDataHeader(UnpooledByteBufAllocator.DEFAULT.buffer());
    }

    @Test(expected = CorruptedBinaryDataRuntimeException.class)
    public void deserializeSmall() {
        new SnapshotLogDataHeader(UnpooledByteBufAllocator.DEFAULT.buffer().writeBytes("junk".getBytes()));
    }

    @Test(expected = CorruptedBinaryDataRuntimeException.class)
    public void deserializeHugeJunk() {
        new SnapshotLogDataHeader(UnpooledByteBufAllocator.DEFAULT.buffer()
            .writeBytes("huge amount of junk that will be instead of original header".getBytes())
            .writeZero(128 << 10));
    }

    private ByteBuf serialize(SnapshotLogDataHeader source) {
        var size = source.size();
        var result = UnpooledByteBufAllocator.DEFAULT.buffer(size, size);
        source.writeTo(result);
        assertEquals(size, result.readableBytes());
        return result;
    }

    private SnapshotLogDataHeader randomHeader() {
        var random = ThreadLocalRandom.current();
        return new SnapshotLogDataHeader(
                CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)],
            random.nextInt(),
            random.nextInt(0, Integer.MAX_VALUE),
            random.nextInt(0, Integer.MAX_VALUE));
    }
}
