package ru.yandex.solomon.slog;

import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.protobuf.MetricType;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMask;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class SnapshotLogDataIteratorImplTest {

    @Parameterized.Parameters(name = "{0}")
    public static Object[] data() {
        return Stream.of(MetricType.values())
                .filter(k -> k != MetricType.UNRECOGNIZED)
                .filter(k -> k != MetricType.METRIC_TYPE_UNSPECIFIED)
                .toArray();
    }

    @Parameterized.Parameter
    public MetricType type;
    private int numId;
    private CompressionAlg compression;

    @Before
    public void setUp() throws Exception {
        var random = ThreadLocalRandom.current();
        numId = random.nextInt();
        compression = CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)];
    }

    @Test
    public void onePoint() {
        var archive = archive();
        archive.addRecord(randomPoint(type));
        test(archive);
    }

    @Test
    public void multiplePoints() {
        var archive = archive();
        int mask = randomMask(type);
        var point = randomPoint(mask);
        archive.addRecord(point);
        for (int index = 0 ; index < 10; index++) {
            randomPoint(point, mask);
            point.tsMillis = archive.getLastTsMillis() + 10_000;
            archive.addRecord(point);
        }
        test(archive);
    }

    @Test
    public void multipleMetrics() {
        var archives = new MetricArchiveMutable[100];
        for (int index = 0; index < archives.length; index++) {
            var archive = archive();
            int mask = randomMask(type);
            archive.addRecord(randomPoint(mask));
            archive.addRecord(randomPoint(mask));
            archive.addRecord(randomPoint(mask));
            archives[index] = archive;
        }
        test(archives);
    }

    private void test(MetricArchiveMutable... archives) {
        ByteBuf buffer = encode(archives);
        try {
            var header = new SnapshotLogDataHeader(buffer);
            assertHeader(header, archives);
            assertData(header, buffer, archives);
        } finally {
            release(buffer);
        }
    }

    private void assertHeader(SnapshotLogDataHeader header, MetricArchiveMutable... archives) {
        assertEquals(numId, header.numId);
        assertEquals(compression, header.compressionAlg);
        assertEquals(archives.length, header.metricsCount);
        var expectedPoints = Stream.of(archives).mapToInt(MetricArchiveMutable::getRecordCount).sum();
        assertEquals(expectedPoints, header.pointsCount);
    }

    private void assertData(SnapshotLogDataHeader header, ByteBuf buffer, MetricArchiveMutable... archives) {
        try (var it = new SnapshotLogDataIteratorImpl(header, buffer)) {
            for (MetricArchiveMutable expected : archives) {
                assertTrue(it.hasNext());
                var actual = new MetricArchiveMutable();
                it.next(actual);
                assertEquals(expected, actual);
            }
            assertFalse(it.hasNext());
        }
    }

    private ByteBuf encode(MetricArchiveMutable... archives) {
        try (var builder = new SnapshotLogDataBuilderImpl(compression, numId, UnpooledByteBufAllocator.DEFAULT)) {
            int packedSize = new SnapshotLogDataHeader(compression, numId).size();
            for (var archive : archives) {
                archive.sortAndMerge();
                packedSize += builder.onTimeSeries(archive.toImmutableNoCopy());
            }

            var buffer = builder.build();
            if (compression == CompressionAlg.NONE) {
                assertEquals(packedSize, buffer.readableBytes());
            }
            return buffer;
        }
    }

    private void release(@Nullable ByteBuf buffer) {
        if (buffer != null) {
            buffer.release();
        }
    }

    private MetricArchiveMutable archive() {
        var archive = new MetricArchiveMutable();
        archive.setType(type);
        archive.setOwnerShardId(numId);
        return archive;
    }
}
