package ru.yandex.solomon.slog;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.Labels;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static ru.yandex.monlib.metrics.MetricType.DGAUGE;
import static ru.yandex.monlib.metrics.MetricType.IGAUGE;

/**
 * @author Vladimir Gordiychuk
 */
public class UnresolvedLogMetaIteratorImplTest {

    private UnresolvedLogMetaBuilder encoder;

    @Before
    public void setUp() {
        var alg = CompressionAlg.values()[ThreadLocalRandom.current().nextInt(CompressionAlg.values().length)];
        var numId = ThreadLocalRandom.current().nextInt();
        encoder = new UnresolvedLogMetaBuilderImpl(numId, alg, UnpooledByteBufAllocator.DEFAULT);
    }

    @After
    public void tearDown() {
        encoder.close();
    }

    @Test
    public void parseOne() {
        var expect = record(DGAUGE, Labels.of("name", "alice"), 1, 14);
        encoder.onCommonLabels(Labels.empty());
        encode(expect);
        var parsed = parse(1, 1);

        assertEquals(1, parsed.size());
        Assert.assertEquals(expect, parsed.get(0));
    }

    @Test
    public void parseEmpty() {
        UnresolvedLogMetaHeader header = new UnresolvedLogMetaHeader(
            42,
            CompressionAlg.LZ4,
            0,
            0,
            0,
            0);

        try (var it = new UnresolvedLogMetaIteratorImpl(header, UnpooledByteBufAllocator.DEFAULT.heapBuffer(), Labels.allocator)) {
            var record = new UnresolvedLogMetaRecord();
            assertFalse(it.next(record));
        }
    }

    @Test
    public void parseFew() {
        var one = record(DGAUGE, Labels.of("name", "alice"), 1, 13);
        var two = record(DGAUGE, Labels.of("name", "bob"), 2, 9);

        encoder.onCommonLabels(Labels.empty());
        encode(one, two);
        var parsed = parse(2, 3);
        assertEquals(2, parsed.size());
        assertEquals(one, parsed.get(0));
        assertEquals(two, parsed.get(1));
    }

    @Test
    public void parseDifferentType() {
        var one = record(DGAUGE, Labels.of("name", "alice"), 1, 13);
        var two = record(IGAUGE, Labels.of("name", "bob"), 1, 9);

        encoder.onCommonLabels(Labels.empty());
        encode(one, two);
        var parsed = parse(2, 2);
        assertEquals(2, parsed.size());
        assertEquals(one, parsed.get(0));
        assertEquals(two, parsed.get(1));
    }

    @Test
    public void parseCommonLabels() {
        var commonLabels = Labels.of("host", "solomon-test-001", "dc", "Sas");
        var one = record(DGAUGE, Labels.of("name", "alice").addAll(commonLabels), 2, 13);
        var two = record(IGAUGE, Labels.of("name", "bob").addAll(commonLabels), 1, 9);

        encoder.onCommonLabels(commonLabels);
        encoder.onMetric(one.type, one.labels.removeByKey("host").removeByKey("dc"), one.points, one.dataSize);
        encoder.onMetric(two.type, two.labels.removeByKey("host").removeByKey("dc"), two.points, two.dataSize);
        var parsed = parse(2, 3);
        assertEquals(2, parsed.size());
        assertEquals(one, parsed.get(0));
        assertEquals(two, parsed.get(1));
    }

    private void encode(UnresolvedLogMetaRecord... records) {
        for (var record : records) {
            encoder.onMetric(record.type, record.labels, record.points, record.dataSize);
        }
    }

    private static UnresolvedLogMetaRecord record(MetricType type, Labels labels, int points, int dataSize) {
        return new UnresolvedLogMetaRecord(type, labels, points, dataSize);
    }

    private List<UnresolvedLogMetaRecord> parse(int expectedMetrics, int expectedPoints) {
        ByteBuf buffer = encoder.build();
        try {
            var header = new UnresolvedLogMetaHeader(buffer);
            assertEquals(expectedMetrics, header.metricsCount);
            assertEquals(expectedPoints, header.pointsCount);
            try (var it = new UnresolvedLogMetaIteratorImpl(header, buffer, Labels.allocator)) {
                List<UnresolvedLogMetaRecord> result = new ArrayList<>(header.metricsCount);
                var record = new UnresolvedLogMetaRecord();
                while (it.next(record)) {
                    result.add(record);
                    record = new UnresolvedLogMetaRecord();
                }
                assertFalse(it.next(record));
                return result;
            }
        } finally {
            buffer.release();
        }
    }
}
