package ru.yandex.solomon.slog.compression;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Consumer;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufUtil;
import io.netty.channel.PreferHeapByteBufAllocator;
import io.netty.channel.unix.PreferredDirectByteBufAllocator;
import org.apache.commons.lang3.RandomStringUtils;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class DecodeStreamTest {
    @Parameterized.Parameter
    public CompressionAlg alg;
    @Parameterized.Parameter(1)
    public boolean heap;

    @Parameterized.Parameters(name = "{0}, heap={1}")
    public static List<Object[]> data() {
        List<Object[]> result = new ArrayList<>();
        for (boolean heap : List.of(true, false)) {
            for (CompressionAlg alg : CompressionAlg.values()) {
                result.add(new Object[]{alg, heap});
            }
        }
        return result;
    }

    @Test
    public void readByte() {
        byte expected = (byte) ThreadLocalRandom.current().nextInt(256);
        check(encode -> encode.writeByte(expected), decode -> {
            var actual = decode.readByte();
            Assert.assertEquals(expected, actual);
        });
    }

    @Test
    public void readIntLe() {
        var expected = ThreadLocalRandom.current().nextInt();
        check(encode -> encode.writeIntLe(expected), decode -> {
            var actual = decode.readIntLe();
            Assert.assertEquals(expected, actual);
        });
    }

    @Test
    public void readLongLe() {
        var expected = ThreadLocalRandom.current().nextLong();
        check(encode -> encode.writeLongLe(expected), decode -> {
            var actual = decode.readLongLe();
            Assert.assertEquals(expected, actual);
        });
    }

    @Test
    public void readDouble() {
        var expected = ThreadLocalRandom.current().nextDouble();
        check(encode -> encode.writeDouble(expected), decode -> {
            var actual = decode.readDoubleLe();
            Assert.assertEquals(expected, actual, 0);
        });
    }

    @Test
    public void readVarint32() {
        var expected = ThreadLocalRandom.current().nextInt();
        check(encode -> encode.writeVarint32(expected), decode -> {
            var actual = decode.readVarint32();
            Assert.assertEquals(expected, actual);
        });
    }

    @Test
    public void readBytesArray() {
        var random = ThreadLocalRandom.current();
        var bytes = random.nextInt(1, 1 << 20); // 1 MiB
        var expected = new byte[bytes];
        random.nextBytes(expected);
        check(encode -> encode.write(expected, 0, bytes), decode -> {
            var actual = new byte[bytes];
            decode.read(actual, 0, bytes);
            assertArrayEquals(expected, actual);
        });
    }

    @Test
    public void readBytesBuffer() {
        var random = ThreadLocalRandom.current();
        var bytes = random.nextInt(1, 1 << 20); // 1 MiB
        var expected = new byte[bytes];
        random.nextBytes(expected);

        check(encode -> encode.write(expected, 0, bytes), decode -> {
            var actual = allocateBuffer();
            try {
                decode.read(actual, bytes);
                assertArrayEquals(expected, ByteBufUtil.getBytes(actual));
            } finally {
                actual.release();
            }
        });
    }

    @Test
    public void readString() {
        var alice = "alice";
        var bob = "bob";
        var time = Instant.now().toString();
        check(
            encode -> {
                encode.writeString(alice);
                encode.writeString(bob);
                encode.writeString(time);
            },
            decode -> {
                Assert.assertEquals("alice", decode.readString());
                Assert.assertEquals("bob", decode.readString());
                Assert.assertEquals(time, decode.readString());
            });
    }

    @Test
    public void readManyStrings() {
        List<String> expected = new ArrayList<>();
        check(
            encode -> {
                for (int index = 0; index < 10_000; index++) {
                    var str = RandomStringUtils.randomAlphanumeric(ThreadLocalRandom.current().nextInt(1, 200));
                    encode.writeString(str);
                    expected.add(str);
                }
            },
            decode -> {
                for (var expect : expected) {
                    Assert.assertEquals(expect, decode.readString());
                }
            }
        );
    }

    @Test
    public void writeBytes2() {
        int frameSize = 64 << 10;
        List<byte[]> expected = new ArrayList<>();
        expected.add(array(frameSize));
        expected.add(array(frameSize - 1));
        expected.add(array(1));
        expected.add(array(frameSize + 1));

        check(
            encode -> {
                for (byte[] bytes : expected) {
                    encode.write(bytes, 0, bytes.length);
                }
            },
            decode -> {
                for (byte[] expect : expected) {
                    var actual = new byte[expect.length];
                    decode.read(actual, 0, actual.length);
                    assertArrayEquals(expect, actual);
                }
            });
    }

    @Test
    public void skipBytes() {
        var random = ThreadLocalRandom.current();
        var bytes = random.nextInt(1, 1 << 20); // 1 MiB
        var noise = new byte[bytes];
        random.nextBytes(noise);
        check(
            encode -> {
                encode.write(noise, 0, bytes);
                encode.writeString("expected");
            },
            decode -> {
                decode.skipBytes(bytes);
                String actual = decode.readString();
                assertEquals("expected", actual);
            });
    }

    private byte[] array(int size) {
        var random = ThreadLocalRandom.current();
        var result = new byte[size];
        random.nextBytes(result);
        return result;
    }

    private ByteBuf allocateBuffer() {
        return allocator().buffer();
    }

    private ByteBufAllocator allocator() {
        return heap ? PreferHeapByteBufAllocator.DEFAULT : PreferredDirectByteBufAllocator.DEFAULT;
    }

    private void check(Consumer<EncodeStream> encodeFn, Consumer<DecodeStream> decodeFn) {
        try (var encode = EncodeStream.create(alg, allocator())) {
            encodeFn.accept(encode);
            var compressed = encode.finishStream();
            try (var decode = DecodeStream.create(alg, compressed)) {
                decodeFn.accept(decode);
            }
        }
    }

    private interface Suit {
        void encode(EncodeStream encoder);

        void decode(DecodeStream decoder);
    }
}
