package ru.yandex.solomon.slog.compression;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.buffer.ByteBufUtil;
import io.netty.channel.PreferHeapByteBufAllocator;
import io.netty.channel.unix.PreferredDirectByteBufAllocator;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.monlib.metrics.encode.spack.compression.DecodeStream;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class EncodeStreamTest {
    @Parameterized.Parameter
    public CompressionAlg alg;
    @Parameterized.Parameter(1)
    public boolean heap;

    private EncodeStream next;
    private ru.yandex.monlib.metrics.encode.spack.compression.EncodeStream prev;
    private ByteBuf prevEncoded;

    @Parameterized.Parameters(name = "{0}, heap={1}")
    public static List<Object[]> data() {
        List<Object[]> result = new ArrayList<>();
        for (boolean heap : List.of(true, false)) {
            for (CompressionAlg alg : CompressionAlg.values()) {
                result.add(new Object[]{alg, heap});
            }
        }
        return result;
    }

    @Before
    public void setUp() throws Exception {
        var allocator = heap ? PreferHeapByteBufAllocator.DEFAULT : PreferredDirectByteBufAllocator.DEFAULT;
        next = EncodeStream.create(alg, allocator);
        prevEncoded = allocateBuffer();
        prev = ru.yandex.monlib.metrics.encode.spack.compression.EncodeStream.create(alg, new ByteBufOutputStream(prevEncoded));
    }

    @After
    public void tearDown() throws Exception {
        prev.close();
        next.close();
        prevEncoded.release();
    }

    @Test
    public void writeByte() {
        byte value = (byte) ThreadLocalRandom.current().nextInt(256);
        next.writeByte(value);
        prev.writeByte(value);

        var buffer = checkBinary();
        try {
            DecodeStream decoder = DecodeStream.create(alg, buffer.nioBuffer());
            assertEquals(value, decoder.readByte());
        } finally {
            buffer.release();
        }
    }

    @Test
    public void writeIntLe() {
        var value = ThreadLocalRandom.current().nextInt();
        next.writeIntLe(value);
        prev.writeIntLe(value);

        var buffer = checkBinary();
        try {
            DecodeStream decoder = DecodeStream.create(alg, buffer.nioBuffer());
            assertEquals(value, decoder.readIntLe());
        } finally {
            buffer.release();
        }
    }

    @Test
    public void writeLongLe() {
        var value = ThreadLocalRandom.current().nextLong();
        next.writeLongLe(value);
        prev.writeLongLe(value);

        var buffer = checkBinary();
        try {
            DecodeStream decoder = DecodeStream.create(alg, buffer.nioBuffer());
            assertEquals(value, decoder.readLongLe());
        } finally {
            buffer.release();
        }
    }

    @Test
    public void writeDouble() {
        var value = ThreadLocalRandom.current().nextDouble();
        next.writeDouble(value);
        prev.writeDouble(value);

        var buffer = checkBinary();
        try {
            DecodeStream decoder = DecodeStream.create(alg, buffer.nioBuffer());
            assertEquals(value, decoder.readDoubleLe(), 0);
        } finally {
            buffer.release();
        }
    }

    @Test
    public void writeVarint32() {
        var value = ThreadLocalRandom.current().nextInt();
        next.writeVarint32(value);
        prev.writeVarint32(value);

        var buffer = checkBinary();
        try {
            DecodeStream decoder = DecodeStream.create(alg, buffer.nioBuffer());
            assertEquals(value, decoder.readVarint32());
        } finally {
            buffer.release();
        }
    }

    @Test
    public void writeBytes() {
        var random = ThreadLocalRandom.current();
        var bytes = random.nextInt(1, 1 << 20); // 1 MiB
        var array = new byte[bytes];
        random.nextBytes(array);
        next.write(array, 0, bytes);
        prev.write(array, 0, bytes);

        var buffer = checkBinary();
        try {
            DecodeStream decoder = DecodeStream.create(alg, buffer.nioBuffer());
            var actual = new byte[bytes];
            decoder.readFully(actual, bytes);
            assertArrayEquals(array, actual);
        } finally {
            buffer.release();
        }
    }

    private ByteBuf allocateBuffer() {
        var allocator = heap ? PreferHeapByteBufAllocator.DEFAULT : PreferredDirectByteBufAllocator.DEFAULT;
        return allocator.buffer();
    }

    private ByteBuf checkBinary() {
        prev.close();
        var result = next.finishStream();
        try {
            assertArrayEquals(ByteBufUtil.getBytes(prevEncoded), ByteBufUtil.getBytes(result));
        } catch (Throwable e) {
            System.out.println("expected");
            System.out.println(ByteBufUtil.prettyHexDump(prevEncoded));
            System.out.println("actual");
            System.out.println(ByteBufUtil.prettyHexDump(result));
            result.release();
            throw new RuntimeException(e);
        }
        return result;
    }
}
