package ru.yandex.solomon.slog.compression.alg;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.buffer.ByteBufUtil;
import org.junit.After;
import org.junit.AssumptionViolatedException;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.monlib.metrics.encode.spack.compression.EncodeStream;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertNotEquals;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class CompressorTest {
    private static final int FRAME_SIZE = 64 << 10;

    ByteBuf expected;
    ByteBuf uncompressed;
    ByteBuf compressed;
    private EncodeStream encodeStream;
    private Compressor compressor;

    @Parameterized.Parameter
    public CompressionAlg alg;
    @Parameterized.Parameter(1)
    public boolean heap;

    @Parameterized.Parameters(name = "{0}, heap={1}")
    public static List<Object[]> data() {
        List<Object[]> result = new ArrayList<>();
        for (boolean heap : List.of(true, false)) {
            for (CompressionAlg alg : CompressionAlg.values()) {
                result.add(new Object[]{alg, heap});
            }
        }
        return result;
    }

    @Before
    public void setUp() throws Exception {
        expected = allocateBuffer();
        uncompressed = allocateBuffer();
        compressed = allocateBuffer();
        encodeStream = EncodeStream.create(alg, new ByteBufOutputStream(expected));
        switch (alg) {
            case LZ4:
                compressor = Lz4Compressor.newInstance();
                break;
            case ZSTD:
                compressor = ZstdCompressor.newInstance();
                break;
            case ZLIB:
                compressor = ZlibCompressor.newInstance();
                break;
            default:
                throw new AssumptionViolatedException("Unknown alg:" + alg);
        }
    }

    private ByteBuf allocateBuffer() {
        if (heap) {
            return ByteBufAllocator.DEFAULT.heapBuffer();
        } else {
            return ByteBufAllocator.DEFAULT.directBuffer();
        }
    }

    @After
    public void tearDown() throws Exception {
        expected.release();
        uncompressed.release();
        compressed.release();
    }

    @Test
    public void writeIntLE() {
        uncompressed.writeIntLE(42);
        encodeStream.writeIntLe(42);
        check();
    }

    @Test
    public void writeLongLe() {
        for (int index = 0; index < 10; index++) {
            long num = ThreadLocalRandom.current().nextLong();
            uncompressed.writeLongLE(num);
            encodeStream.writeLongLe(num);
        }
        check();
    }

    @Test
    public void writeDouble() {
        for (int index = 0; index < 10; index++) {
            double num = ThreadLocalRandom.current().nextDouble();
            uncompressed.writeDoubleLE(num);
            encodeStream.writeDouble(num);
        }
        check();
    }

    @Test
    public void writeOneNotFullFrame() {
        int frameSize = FRAME_SIZE - 42;
        byte[] bytes = new byte[frameSize];
        ThreadLocalRandom.current().nextBytes(bytes);
        encodeStream.write(bytes, 0, frameSize);
        uncompressed.writeBytes(bytes, 0, frameSize);
        compressor.compress(uncompressed, compressed);
        uncompressed.clear();
        encodeStream.close();
        compressor.finish(compressed);
        assertEqual();
    }

    @Test
    public void writeOneFullFrame() {
        int frameSize = FRAME_SIZE;
        byte[] bytes = new byte[frameSize];
        ThreadLocalRandom.current().nextBytes(bytes);
        encodeStream.write(bytes, 0, frameSize);
        uncompressed.writeBytes(bytes, 0, frameSize);
        compressor.compress(uncompressed, compressed);
        uncompressed.clear();
        encodeStream.close();
        compressor.finish(compressed);
        assertEqual();
    }

    @Test
    public void writeBytes() {
        int frameSize = FRAME_SIZE;
        int frames = 10;
        byte[] bytes = new byte[frameSize * frames];
        ThreadLocalRandom.current().nextBytes(bytes);
        for (int index = 0; index < frames; index++) {
            int offset = index * frameSize;
            encodeStream.write(bytes, offset, frameSize);
            uncompressed.writeBytes(bytes, offset, frameSize);
            compressor.compress(uncompressed, compressed);
            uncompressed.clear();
        }

        encodeStream.close();
        compressor.finish(compressed);
        assertEqual();
    }

    private void check() {
        encodeStream.close();
        compressor.compress(uncompressed, compressed);
        int compressedSize = compressed.readableBytes();
        assertNotEquals("Not able compress to 0 bytes",0, compressedSize);

        uncompressed.clear();
        compressor.finish(compressed);
        assertNotEquals("Should be add empty frame", compressedSize, compressed.readableBytes());
        assertEqual();
    }

    private void assertEqual() {
        try {
            assertArrayEquals(ByteBufUtil.getBytes(expected), ByteBufUtil.getBytes(compressed));
        } catch (Throwable e) {
            System.out.println("expected");
            System.out.println(ByteBufUtil.prettyHexDump(expected));
            System.out.println("actual");
            System.out.println(ByteBufUtil.prettyHexDump(compressed));
            throw new RuntimeException(e);
        }
    }
}
