package ru.yandex.solomon.slog.compression.alg;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufUtil;
import org.junit.After;
import org.junit.AssumptionViolatedException;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;

import static org.junit.Assert.assertArrayEquals;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class DecompressorTest {
    ByteBuf actual;
    ByteBuf expected;
    ByteBuf uncompressed;
    ByteBuf compressed;
    private Compressor compressor;
    private Decompressor decompressor;

    @Parameterized.Parameter
    public CompressionAlg alg;
    @Parameterized.Parameter(1)
    public boolean heap;

    @Parameterized.Parameters(name = "{0}, heap={1}")
    public static List<Object[]> data() {
        List<Object[]> result = new ArrayList<>();
        for (boolean heap : List.of(true, false)) {
            for (CompressionAlg alg : CompressionAlg.values()) {
                result.add(new Object[]{alg, heap});
            }
        }
        return result;
    }

    @Before
    public void setUp() throws Exception {
        actual = allocateBuffer();
        expected = allocateBuffer();
        uncompressed = allocateBuffer();
        compressed = allocateBuffer();
        switch (alg) {
            case LZ4:
                compressor = Lz4Compressor.newInstance();
                decompressor = Lz4Decompressor.newInstance();
                break;
            case ZSTD:
                compressor = ZstdCompressor.newInstance();
                decompressor = ZstdDecompressor.newInstance();
                break;
            case ZLIB:
                compressor = ZlibCompressor.newInstance();
                decompressor = ZlibDecompressor.newInstance();
                break;
            default:
                throw new AssumptionViolatedException("Unknown alg:" + alg);
        }
    }


    @After
    public void tearDown() throws Exception {
        uncompressed.release();
        compressed.release();
        expected.release();
        actual.release();
    }

    @Test
    public void num() {
        uncompressed.writeIntLE(42);
        expected.writeIntLE(42);

        compressor.compress(uncompressed, compressed);
        decompressor.decompress(compressed, actual);
        assertBuffersEqual(expected, actual);
    }

    @Test
    public void randomOneChunk() {
        var random = ThreadLocalRandom.current();
        var size = random.nextInt(1, 64 << 10);
        var bytes = new byte[size];
        random.nextBytes(bytes);

        uncompressed.writeBytes(bytes);
        expected.writeBytes(bytes);

        compressor.compress(uncompressed, compressed);
        decompressor.decompress(compressed, actual);
        assertBuffersEqual(expected, actual);
    }

    @Test
    public void randomMultipleChunk() {
        var random = ThreadLocalRandom.current();
        var count = 10;

        for (int index = 0; index < count; index++) {
            var size = random.nextInt(1, 64 << 10);
            var bytes = new byte[size];
            random.nextBytes(bytes);
            uncompressed.writeBytes(bytes);
            expected.writeBytes(bytes);
            compressor.compress(uncompressed, compressed);
            uncompressed.clear();
        }

        for (int index = 0; index < count; index++) {
            decompressor.decompress(compressed, actual);
        }
        assertBuffersEqual(expected, actual);
    }

    private ByteBuf allocateBuffer() {
        if (heap) {
            return ByteBufAllocator.DEFAULT.heapBuffer();
        } else {
            return ByteBufAllocator.DEFAULT.directBuffer();
        }
    }

    private void assertBuffersEqual(ByteBuf expected, ByteBuf actual) {
        try {
            assertArrayEquals(ByteBufUtil.getBytes(expected), ByteBufUtil.getBytes(actual));
        } catch (Throwable e) {
            System.out.println("expected");
            System.out.println(ByteBufUtil.prettyHexDump(expected));
            System.out.println("actual");
            System.out.println(ByteBufUtil.prettyHexDump(actual));
            throw new RuntimeException(e);
        }
    }
}
