package ru.yandex.solomon.spring;

import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.HierarchicalBeanFactory;
import org.springframework.beans.factory.ListableBeanFactory;
import org.springframework.context.annotation.Condition;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.context.annotation.ConfigurationCondition;
import org.springframework.core.annotation.AnnotationAttributes;
import org.springframework.core.type.AnnotatedTypeMetadata;

/**
 * @author Sergey Polovko
 */
public class OnBeanCondition implements Condition, ConfigurationCondition {
    @Override
    public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
        if (metadata.isAnnotated(ConditionalOnBean.class.getName())) {
            var attributes = AnnotationAttributes.fromMap(
                    metadata.getAnnotationAttributes(ConditionalOnBean.class.getName()));
            return hasBean(context, attributes.getClass("value"));
        }

        if (metadata.isAnnotated(ConditionalOnMissingBean.class.getName())) {
            var attributes = AnnotationAttributes.fromMap(
                    metadata.getAnnotationAttributes(ConditionalOnMissingBean.class.getName()));
            return !hasBean(context, attributes.getClass("value"));
        }

        return false;
    }

    @Override
    public ConfigurationPhase getConfigurationPhase() {
        return ConfigurationPhase.REGISTER_BEAN;
    }

    private static boolean hasBean(ConditionContext context, Class<?> value) {
        ListableBeanFactory beanFactory = context.getBeanFactory();
        while (true) {
            String[] beans = beanFactory.getBeanNamesForType(value, true, false);
            if (beans.length != 0) {
                return true;
            }

            if (beanFactory instanceof HierarchicalBeanFactory) {
                BeanFactory parent = ((HierarchicalBeanFactory) beanFactory).getParentBeanFactory();
                if (parent instanceof ListableBeanFactory) {
                    beanFactory = (ListableBeanFactory) parent;
                } else {
                    break;
                }
            }
        }

        return false;
    }
}
