package ru.yandex.solomon.staffOnly;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.staffOnly.manager.flamegraph.FlameGraphProducer;

/**
 * @author Ivan Tsybulin
 */
@RestController
@Configuration
@ParametersAreNonnullByDefault
@Import({
        FlameGraphContext.class
})
public class FlameGraphController {
    private static final Logger logger = LoggerFactory.getLogger(FlameGraphController.class);
    @Autowired
    HttpAuthenticator authenticator;
    @Autowired
    InternalAuthorizer authorizer;
    @Autowired
    FlameGraphProducer flameGraphProducer;

    @Bean
    public RootLink flameGraphLink() {
        return new RootLink("/flameGraph", "Flame graph");
    }

    @RequestMapping(value = "/flameGraph", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> flameGraph(
            @RequestParam Map<String, String> params,
            ServerHttpRequest request)
    {
        return authorize(request)
                .thenApply(account -> {
                    applyParams(params);
                    return new FlameGraphPage(flameGraphProducer).genString();
                })
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("/flameGraph failed", e);
                    }
                });
    }

    @RequestMapping("/flameGraph/start")
    public CompletableFuture<ResponseEntity<String>> start(
            @RequestParam Map<String, String> params,
            ServerHttpRequest request)
    {
        return authorize(request)
                .thenAccept(account -> {
                    applyParams(params);
                    flameGraphProducer.start();
                })
                .thenApply(ignore -> redirectBack())
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("/flameGraph/start failed", e);
                    }
                });
    }

    private void applyParams(Map<String, String> params) {
        String threadNameRegex = params.get("threadNameRegex");
        if (threadNameRegex != null) {
            flameGraphProducer.setThreadNameRegex(threadNameRegex);
        }

        String depth = params.get("depth");
        if (depth != null) {
            flameGraphProducer.setMaxStackDepth(Integer.parseInt(depth));
        }

        String dropFramePercent = params.get("dropFramePercent");
        if (dropFramePercent != null) {
            flameGraphProducer.setDropFramesPercent(Double.parseDouble(dropFramePercent));
        }
    }

    @RequestMapping("/flameGraph/stop")
    public CompletableFuture<ResponseEntity<String>> stop(
            @RequestParam Map<String, String> params,
            ServerHttpRequest request)
    {
        return authorize(request)
                .thenAccept(account -> {
                    applyParams(params);
                    flameGraphProducer.stop();
                })
                .thenApply(ignore -> redirectBack())
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("/flameGraph/stop failed", e);
                    }
                });
    }

    @RequestMapping("/flameGraph/refresh")
    public CompletableFuture<ResponseEntity<String>> refresh(
            @RequestParam Map<String, String> params,
            ServerHttpRequest request)
    {
        return authorize(request)
                .thenAccept(account -> {
                    applyParams(params);
                    flameGraphProducer.refresh();
                })
                .thenApply(ignore -> redirectBack())
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("/flameGraph/refresh failed", e);
                    }
                });
    }

    private ResponseEntity<String> redirectBack() {
        return ResponseEntity.status(HttpStatus.FOUND)
                .header("Location", "/flameGraph")
                .body(null);
    }

    private CompletableFuture<Account> authorize(ServerHttpRequest request) {
        return authenticator.authenticate(request)
                .thenCompose(authSubject -> authorizer.authorize(authSubject));
    }

}
