package ru.yandex.solomon.staffOnly;

import java.time.Duration;
import java.time.Instant;
import java.util.EnumMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.commons.lang3.StringEscapeUtils;

import ru.yandex.solomon.staffOnly.manager.flamegraph.FlameGraphProducer;
import ru.yandex.solomon.staffOnly.manager.flamegraph.Trie;
import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Ivan Tsybulin
 */
class FlameGraphPage extends ManagerPageTemplate {

    private static final EnumMap<FlameGraphProducer.State, String> STATE_COLORS = new EnumMap<>(Map.of(
            FlameGraphProducer.State.EMPTY, "default",
            FlameGraphProducer.State.RUNNING, "success",
            FlameGraphProducer.State.DONE, "info"
    ));
    private static final Map<String, String> BUTTON_COLORS = Map.of(
            "Start", "success",
            "Stop", "danger",
            "Refresh", "info"
    );
    private final FlameGraphProducer flameGraphProducer;

    public FlameGraphPage(FlameGraphProducer flameGraphProducer) {
        super("FlameGraph");
        this.flameGraphProducer = flameGraphProducer;
    }

    private static String sinceTime(Instant time) {
        if (time.equals(Instant.EPOCH)) {
            return "never";
        }
        long millis = Duration.between(time, Instant.now()).toMillis();
        String duration = DurationUtils.formatDurationMillisTruncated(millis);
        if (duration.equals("0")) {
            return "now";
        }
        return duration + " ago";
    }

    private void textInput(String id, String name, String value, String help) {
        tr(() -> {
            thText(name);
            td(() -> inputTextfieldFormControl(id, id, value));
            td(() -> tag("span", Attr.cssClass("help-block"), () -> write(help)));
        });
    }

    @Override
    protected void content() {
        var stateChanged = flameGraphProducer.getStateChanged();
        var state = stateChanged.currentState();
        var since = stateChanged.since();

        form(() -> {
            inputGroup(() ->
                tag("div.form-group", () -> {
                    textInput("threadNameRegex", "Thread name regex:",
                            flameGraphProducer.getThreadNameRegex(), "(e.g. CpuLowPriority-.*)");
                    textInput("depth", "Max stack depth:",
                            Integer.toString(flameGraphProducer.getMaxStackDepth()), "");
                    textInput("dropFramePercent", "Drop frames threshold:",
                            String.format(Locale.ROOT, "%.3g", flameGraphProducer.getDropFramesPercent()),
                            "drop frame if its percentage to parent is less than given");
                    textInput("samplingFreq", "Sampling frequency:",
                            Integer.toString(flameGraphProducer.getSamplingFrequency()),
                            "keep below 1000");
                })
            );

            tag("div.col-xs-4.btn-group.pull-right", () -> {
                for (var action : List.of("Start", "Stop", "Refresh")) {
                    tag("button.btn.btn-" + BUTTON_COLORS.get(action) + " type=submit", () -> write(action),
                            new Attr("formaction", "/flameGraph/" + action.toLowerCase(Locale.ROOT)));
                }
            });

        }, new Attr("action", ""));

        tag("div.row", () ->
            tag("div.col-xs-8", () -> {
                write("State: ");
                tag("span", new Attr("class", "label label-" + STATE_COLORS.get(state)),
                        () -> write(state.name()));
                write(", changed: " + sinceTime(since));
            })
        );

        var flameGraphWithTime = flameGraphProducer.get();
        var flameGraph = flameGraphWithTime.flameGraph();
        var time = flameGraphWithTime.time();
        tag("div", () -> write("FlameGraph taken " + sinceTime(time)));

        tag("div#chart");
        blockTag("script", () -> {},
                new Attr("type", "text/javascript"),
                new Attr("src", "https://yastatic.net/d3/4.5.0/d3.min.js"));
        blockTag("script", () -> {},
                new Attr("type", "text/javascript"),
                new Attr("src", "/static/d3-flamegraph/d3-flamegraph.min.js"));
        linkStylesheet("/static/d3-flamegraph/d3-flamegraph.css");
        blockTag("script", () -> {
            writeRaw("var chart = flamegraph();\n");
            writeRaw("var data = ");
            writeRaw(convertToJson("root", flameGraph));
            writeRaw(";\n");
            writeRaw("d3.select(\"#chart\").datum(data).call(chart);\n");
        }, new Attr("type", "text/javascript"));
    }

    private String convertToJson(String nodeName, Trie.ImmutableTrieNode node) {
        StringBuilder sb = new StringBuilder();
        sb.append("{\"value\":");
        sb.append(node.counter());
        sb.append(",\"name\":\"");
        sb.append(StringEscapeUtils.escapeJson(nodeName));
        sb.append("\"");
        var children = node.children();
        if (children.isEmpty()) {
            return sb.append("}").toString();
        }
        sb.append(",\"children\":[");
        String delim = "";
        long dropChildrenThreshold = (long)(node.counter() * flameGraphProducer.getDropFramesPercent() / 100d);
        for (var e : children.entrySet()) {
            long childValue = e.getValue().counter();
            if (childValue < dropChildrenThreshold) {
                continue;
            }
            sb.append(delim);
            delim = ",";
            sb.append(convertToJson(e.getKey(), e.getValue()));
        }
        return sb.append("]}").toString();
    }
}
