package ru.yandex.solomon.staffOnly;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.AppenderRef;
import org.apache.logging.log4j.core.config.LoggerConfig;
import org.apache.logging.log4j.core.config.Property;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
@Configuration
@ParametersAreNonnullByDefault
public class LoggersController {
    private static final Logger logger = LoggerFactory.getLogger(LoggersController.class);
    @Autowired
    HttpAuthenticator authenticator;
    @Autowired
    InternalAuthorizer authorizer;

    @Bean
    public RootLink loggersLink() {
        return new RootLink("/loggers", "Loggers");
    }

    @RequestMapping(value = "/loggers", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> loggers(ServerHttpRequest request, @RequestParam Map<String, String> params) {
        return authorize(request)
                .thenApply(account -> new LoggersPage(params).genString())
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("/loggers failed", e);
                    }
                });
    }

    @RequestMapping("/loggers/change")
    public CompletableFuture<ResponseEntity<String>> changeLogger(
            @RequestParam(value = "name") String name,
            @RequestParam(value = "level", defaultValue = "INFO") String level,
            ServerHttpRequest request)
    {
        return authorize(request)
                .thenAccept(account -> {
                    Level targetLevel = Level.toLevel(level, Level.INFO);
                    LoggerConfig loggerConfig = getLoggerConfig(name);
                    if (loggerConfig == null) {
                        var root = getLoggerContext().getRootLogger();
                        var rootConf = root.get();
                        loggerConfig = LoggerConfig.createLogger(
                                root.isAdditive(),
                                targetLevel,
                                name,
                                Boolean.toString(rootConf.isIncludeLocation()),
                                Nullables.orEmpty(rootConf.getAppenderRefs()).toArray(AppenderRef[]::new),
                                Nullables.orEmpty(rootConf.getPropertyList()).toArray(Property[]::new),
                                getLoggerContext().getConfiguration(),
                                rootConf.getFilter());
                        for (var appender : rootConf.getAppenders().values()) {
                            loggerConfig.addAppender(appender, null, null);
                        }

                        getLoggerContext().getConfiguration().addLogger(name, loggerConfig);
                    } else {
                        loggerConfig.setLevel(targetLevel);
                    }
                    getLoggerContext().updateLoggers();
                })
                .thenApply(ignore -> redirectBack())
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("/loggers/change failed", e);
                    }
                });
    }

    private LoggerConfig getLoggerConfig(String name) {
        var config = getLoggerContext().getConfiguration();
        if (name.equals("root") || Strings.isNullOrEmpty(name)) {
            return config.getRootLogger();
        }
        return config.getLoggers().get(name);
    }

    private LoggerContext getLoggerContext() {
        return LoggerContext.getContext(false);
    }

    private ResponseEntity<String> redirectBack() {
        return ResponseEntity.status(HttpStatus.FOUND)
                .header("Location", "/loggers")
                .body(null);
    }

    private CompletableFuture<Account> authorize(ServerHttpRequest request) {
        return authenticator.authenticate(request)
                .thenCompose(authSubject -> authorizer.authorize(authSubject));
    }

}
