package ru.yandex.solomon.staffOnly;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Strings;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.core.Logger;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.LoggerConfig;
import org.apache.logging.log4j.spi.AbstractLogger;

import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
class LoggersPage extends ManagerPageTemplate {
    public LoggersPage(Map<String, String> params) {
        super("Loggers");
    }

    @Override
    protected void content() {
        var context = LoggerContext.getContext(false);
        var configuration = context.getConfiguration();
        h2("Configs");
        tableTable(() -> {
            trThs("Name", "Level");
            writeLoggerRecord(configuration.getRootLogger());
            for (var logger : configuration.getLoggers().values()) {
                writeLoggerRecord(logger);
            }
            writeFormAddNewLogger();
        });

        h2("Loggers");
        tableTable(() -> {
            trThs("Name", "Parent", "Level");
            context.getLoggers()
                    .stream()
                    .filter(logger -> !Strings.isNullOrEmpty(logger.getName()))
                    .sorted(Comparator.comparing(AbstractLogger::getName))
                    .forEach(this::writeLoggerRecord);
        });
    }

    private void writeLoggerRecord(LoggerConfig logger) {
        tr(() -> {
            String name = formatLoggerName(logger.getName());
            tdText(name);
            td(() -> {
                for (var level : Level.values()) {
                    String type = level != logger.getLevel()
                            ? "secondary"
                            : "primary";

                    String href = "/loggers/change?level=" + level.name() + "&name=" + URLEncoder.encode(name, StandardCharsets.UTF_8);
                    tag("a.btn.btn-" + type + " role=button", () -> write(level.name()), new Attr("href", href));
                }
            });
        });
    }

    private String formatLoggerName(String name) {
        return Nullables.orDefault(Strings.emptyToNull(name), "root");
    }

    private void writeLoggerRecord(Logger logger) {
        if (Objects.equals(logger.getName(), logger.get().getName())) {
            return;
        }

        tr(() -> {
            String name = formatLoggerName(logger.getName());
            tdText(formatLoggerName(name));
            tdText(formatLoggerName(logger.get().getName()));
            td(() -> {
                for (var level : Level.values()) {
                    String type = level != logger.getLevel()
                            ? "secondary"
                            : "primary";

                    String href = "/loggers/change?level=" + level.name() + "&name=" + URLEncoder.encode(name, StandardCharsets.UTF_8);
                    tag("a.btn.btn-" + type + " role=button", () -> write(level.name()), new Attr("href", href));
                }
            });
        });
    }

    private void writeFormAddNewLogger() {
        form(() -> {
            tr(() -> {
                td(() -> {
                    tag("div.btn-group", () -> {
                        inputTextfieldFormControl("logger", "name", "");
                    });
                });
                td(() -> {
                    List<String> fqdns = Stream.of(Level.values())
                            .map(Level::name)
                            .collect(Collectors.toList());
                    tag("div.btn-group", () -> {
                        selectFormControl("level", "level", fqdns, "");
                    });
                    tag("div.btn-group", () -> {
                        tag("button.btn.btn-default type=submit formaction=/loggers/change", () -> write("Add"));
                    });
                });
            });
        });
    }
}
