package ru.yandex.solomon.staffOnly;

import java.lang.management.ManagementFactory;
import java.time.Instant;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.staffOnly.manager.ManagerController;
import ru.yandex.solomon.staffOnly.manager.ok.OkController;
import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.SolomonEnv;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.solomon.util.http.HttpUtils;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Stepan Koltsov
 */
@RestController
@Configuration
@Import({
    ManagerController.class,
    WtdFilterContext.class,
    OkController.class,
    LoggersController.class,
    FlameGraphController.class,
    NettyMemoryPage.class,
})
@ParametersAreNonnullByDefault
public class MiscController {
    @Autowired
    HttpAuthenticator authenticator;
    @Autowired
    InternalAuthorizer authorizer;

    @Bean
    public RootLink threadsLink() {
        return new RootLink("/threads", "Threads");
    }

    // It places here to prevent cyclic dependency beteween solomon-gateway-core and solomon-staffOnly.
    // It's supposed that every service has /metrics endpoint.
    @Bean
    public RootLink textMetrics() {
        return new RootLink("/metrics?format=TEXT", "My Metrics");
    }

    @RequestMapping(value = "/threads", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> threads(ServerHttpRequest request, @RequestParam Map<String, String> params) {
        return authenticator.authenticate(request)
            .thenCompose(authSubject -> authorizer.authorize(authSubject))
            .thenApply(account -> new ThreadsPage("/threads", params).genString());
    }

    @Bean
    public RootLink versionLink() {
        return new RootLink("/version", "Version");
    }

    @Autowired(required = false)
    private VersionPageProperties[] versionPageProperties;

    @RequestMapping(value = "/version", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> version(ServerHttpRequest request) {
        return authenticator.authenticate(request)
            .thenCompose(authSubject -> authorizer.authorize(authSubject))
            .thenApply(account -> versionImpl());
    }

    public static class DebugDto {
        @JsonProperty
        public Map<String, String> headers;

        @JsonProperty
        public String authType;

        @JsonProperty
        public String authId;

        @JsonProperty
        public String remoteHost;
    }

    @RequestMapping(value = "/debug", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<DebugDto> debug(ServerHttpRequest request) {
        return authenticator.authenticate(request)
            .thenCompose(authSubject -> authorizer.authorize(authSubject))
            .thenApply(account -> {
                DebugDto dto = new DebugDto();
                dto.headers = request.getHeaders().toSingleValueMap();
                dto.authId = account.getId();
                dto.authType = account.getAuthType().name();
                dto.remoteHost = HttpUtils.realOrRemoteIp(request);
                return dto;
            });
    }

    private String versionImpl() {
        return new ManagerPageTemplate("Version") {
            @Override
            protected void content() {
                pre(() -> {
                    String versionRaw = SolomonVersion.readRaw();
                    if (!versionRaw.endsWith("\n")) {
                        versionRaw += "\n";
                    }
                    write(versionRaw);
                    write("\n");

                    Instant startTime = Instant.ofEpochMilli(ManagementFactory.getRuntimeMXBean().getStartTime());

                    Tuple2List<String, String> properties = Tuple2List.arrayList();
                    properties.add("started", InstantUtils.toSecondsFormatter.format(startTime));
                    properties.add("hostname", HostUtils.getFqdn());
                    properties.add("environment", SolomonEnv.current().name());

                    if (versionPageProperties != null) {
                        for (VersionPageProperties ps : versionPageProperties) {
                            properties.addAll(ps.getProperties());
                        }
                    }

                    write(formatKvTable(properties));

                    // TODO: stockpile has no env type
                    //write("env type: " + envType + "\n");
                });

            }
        }.genString();
    }

    static String formatKvTable(Tuple2List<String, String> kvTable) {
        if (kvTable.isEmpty()) {
            return "";
        }
        int firstColumnWidth = kvTable.get1().stream().mapToInt(String::length).max().getAsInt();
        StringBuilder sb = new StringBuilder();
        for (Tuple2<String, String> kv : kvTable) {
            sb.append(StringUtils.rightPad(kv._1 + ":", firstColumnWidth + 1));
            sb.append(" ");
            sb.append(kv._2);
            sb.append("\n");
        }

        return sb.toString();
    }
}
