package ru.yandex.solomon.staffOnly;

import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import io.netty.buffer.AbstractByteBuf;
import io.netty.buffer.ByteBuf;
import io.netty.util.ResourceLeakDetector;
import io.netty.util.ResourceLeakDetector.Level;

import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethodArgument;
import ru.yandex.solomon.staffOnly.manager.ExtraContentParam;
import ru.yandex.solomon.staffOnly.manager.special.ExtraContent;

/**
 * @author Vladimir Gordiychuk
 */
@LinkedOnRootPage("Netty Memory")
public class NettyMemoryPage {

    private final ResourceLeakDetector<ByteBuf> resourceLeakDetectorByteBuf;

    public NettyMemoryPage() {
        this.resourceLeakDetectorByteBuf = resourceLeakDetectorByteBuf();
    }

    @ManagerMethod
    public Level getLevel() {
        return io.netty.util.ResourceLeakDetector.getLevel();
    }

    @ManagerMethod
    public void setLevel(@ManagerMethodArgument(name = "level") Level level) {
        ResourceLeakDetector.setLevel(level);
    }

    @ExtraContent("Leak detection level")
    public void extraDetectionLevel(ExtraContentParam p) {
        p.getHtmlWriter().preText("-Dio.netty.leakDetection.level=" + getLevel());
    }

    @ExtraContent("Reported leaks")
    public void extraText(ExtraContentParam p) {
        var leakDetector = resourceLeakDetectorByteBuf();
        var leaks = reportedLeaks(leakDetector);
        var writer = p.getHtmlWriter();
        if (leaks.isEmpty()) {
            writer.preText("None");
            return;
        }

        for (var stack : leaks) {
            writer.preText(stack);
        }
    }

    @ExtraContent("Tracked buffers")
    public void aliveText(ExtraContentParam p) {
        var allLeaks = allLeaks(resourceLeakDetectorByteBuf());
        var writer = p.getHtmlWriter();
        var uniq = allLeaks.stream()
                .map(Object::toString)
                .filter(s -> !s.isEmpty())
                .collect(Collectors.groupingBy(Function.identity(), Collectors.counting()))
                .entrySet()
                .stream()
                .sorted((o1, o2) -> Long.compare(o1.getValue(), o2.getValue()) * -1)
                .collect(Collectors.toList());

        if (uniq.isEmpty()) {
            writer.preText("None");
            return;
        }

        for (var entry : uniq) {
            writer.preText(entry.getValue() + " count:\n" + entry.getKey());
        }
    }

    static ResourceLeakDetector<ByteBuf> resourceLeakDetectorByteBuf() {
        try {
            var field = AbstractByteBuf.class.getDeclaredField("leakDetector");
            field.setAccessible(true);
            return (ResourceLeakDetector<ByteBuf>) field.get(null);
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    static Set<String> reportedLeaks(ResourceLeakDetector<?> leakDetector) {
        try {
            var field = ResourceLeakDetector.class.getDeclaredField("reportedLeaks");
            field.setAccessible(true);
            return (Set<String>) field.get(leakDetector);
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    static Set<Object> allLeaks(ResourceLeakDetector<?> leakDetector) {
        try {
            var field = ResourceLeakDetector.class.getDeclaredField("allLeaks");
            field.setAccessible(true);
            return (Set<Object>) field.get(leakDetector);
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }
}
