package ru.yandex.solomon.staffOnly;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableList;
import org.springframework.beans.factory.ListableBeanFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.manager.ManagerController;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectFindContext;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.www.IndexPageTemplate;
import ru.yandex.solomon.staffOnly.www.Link;

import static org.springframework.beans.factory.BeanFactoryUtils.beansOfTypeIncludingAncestors;


/**
 * @author Stepan Koltsov
 */
@RestController
@Configuration
@Import({
    MiscController.class,
    ManagerController.class,
    RedirectExceptionHandler.class,
})
public class StaffOnlyController {

    private final ListableBeanFactory beanFactory;
    private final HttpAuthenticator authenticator;
    private final InternalAuthorizer authorizer;
    private final ImmutableList<RootLink> rootLinks;

    @Autowired
    public StaffOnlyController(
            ListableBeanFactory beanFactory,
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer)
    {
        this.beanFactory = beanFactory;
        this.authenticator = authenticator;
        this.authorizer = authorizer;

        var rootLinks = new ArrayList<>(beansOfTypeIncludingAncestors(beanFactory, RootLink.class).values());
        for (var root : beansOfTypeIncludingAncestors(beanFactory, RootLinks.class).values()) {
            rootLinks.addAll(root.getLinks());
        }

        for (Map.Entry<String, Object> e : beansOfTypeIncludingAncestors(beanFactory, null, false, false).entrySet()) {
            String beanName = e.getKey();
            Object bean = e.getValue();
            if (bean == null) {
                continue;
            }
            Option<LinkedOnRootPage> linkedOnRootPage = ClassX.wrap(bean.getClass())
                .getAnnotationInClassOrSupers(LinkedOnRootPage.class);
            if (linkedOnRootPage.isPresent()) {
                NamedObjectId objectId = new NamedObjectId(NamedObjectFindContext.BEAN_CLASS_ID, beanName);
                String title = linkedOnRootPage.get().value();
                rootLinks.add(new RootLink(ManagerController.namedObjectLink(objectId), title));
            }
        }

        rootLinks.sort(Comparator.comparing(RootLink::getTitle));
        this.rootLinks = ImmutableList.copyOf(rootLinks);
    }

    @RequestMapping(value = "/internal", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> internal(ServerHttpRequest request) {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenApply(account -> internalImpl());
    }

    private String internalImpl() {
        return new IndexPageTemplate(JavaProcess.MAIN_CLASS) {
            @Override
            protected List<Link> links() {
                return rootLinks.stream().map(RootLink::toLink).collect(Collectors.toList());
            }
        }.genString();
    }
}
