package ru.yandex.solomon.staffOnly;

import java.lang.management.ManagementFactory;
import java.lang.management.ThreadInfo;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.Map;

import org.apache.commons.lang3.mutable.MutableInt;

import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.solomon.util.collection.enums.EnumMapToLong;

/**
 * @author Sergey Polovko
 */
class ThreadsPage extends ManagerPageTemplate {

    private static final String NAME_FILTER_PARAM = "t";
    private static final String STATE_FILTER_PARAM = "s";
    private static final String COLLAPSE_FILTER_PARAM = "c";

    private final Map<String, String> params;
    private final String pagePath;
    private final String nameFilter;
    private final boolean collapse;
    private final Thread.State stateFilter;

    ThreadsPage(String pagePath, Map<String, String> params) {
        super("Threads");
        this.params = params;
        this.pagePath = pagePath;
        this.nameFilter = params.getOrDefault(NAME_FILTER_PARAM, "").toUpperCase();
        this.collapse = !params.getOrDefault(COLLAPSE_FILTER_PARAM, "y").equals("n");

        String stateFilterStr = params.get(STATE_FILTER_PARAM);
        this.stateFilter = (stateFilterStr == null) ? null : Thread.State.valueOf(stateFilterStr.toUpperCase());
    }

    @Override
    protected void content() {
        ThreadInfo[] threadInfos = ManagementFactory.getThreadMXBean().dumpAllThreads(false, false);
        Arrays.sort(threadInfos, Comparator.comparing(ThreadInfo::getThreadName));

        var threadsPerState = new EnumMapToLong<>(Thread.State.class);
        for (ThreadInfo threadInfo : threadInfos) {
            threadsPerState.incrementAndGet(threadInfo.getThreadState());
        }

        tag("div.btn-toolbar", new Attr("style", "margin-bottom: 40px"), () -> {
            tag("div.btn-group", () -> {
                String href = hrefWithParam(COLLAPSE_FILTER_PARAM, collapse ? "n" : "y");
                tag("a.btn.btn-default", new Attr("href", href), () -> {
                    tag("span.glyphicon.glyphicon-resize-" + (collapse ? "full" : "small"));
                });
            });
            tag("div.btn-group", () -> {
                for (Thread.State state : Thread.State.values()) {
                    long count = threadsPerState.get(state);
                    if (count == 0) {
                        continue;
                    }

                    String tag = (stateFilter == state) ? "a.btn.btn-default.active" : "a.btn.btn-default";
                    String href = hrefWithParam(STATE_FILTER_PARAM, state.name());
                    tag(tag, new Attr("href", href), () -> {
                        write(state.name() + ' ');
                        tag("span.badge", () -> write(count));
                    });
                }
            });
            tag("div.btn-group", () -> {
                form(() -> {
                        tag("input",
                            new Attr("type", "text"),
                            new Attr("name", NAME_FILTER_PARAM),
                            new Attr("value", nameFilter),
                            new Attr("class", "form-control"),
                            new Attr("placeholder", "name filter"));
                        params.forEach((pn, pv) -> {
                            if (!pn.equals(NAME_FILTER_PARAM)) {
                                inputHidden(pn, pv);
                            }
                        });
                    },
                    new Attr("action", pagePath));
            });
            tag("div.btn-group.pull-right", () -> {
                tag("a.btn.btn-warning", new Attr("href", pagePath), () -> write("Clear Filters"));
            });
        });

        MutableInt threadId = new MutableInt(1);
        for (ThreadInfo thread : threadInfos) {
            if (!nameFilter.isEmpty()) {
                if (!thread.getThreadName().toUpperCase().contains(nameFilter)) {
                    continue;
                }
            }
            if (stateFilter != null) {
                if (thread.getThreadState() != stateFilter) {
                    continue;
                }
            }

            tag("div.row", () -> {
                tag("div.col-md-4", () -> {
                    tag("a", () -> h4(thread.getThreadName()),
                        new Attr("href", "#" + threadId.intValue()),
                        new Attr("data-toggle", "collapse"));
                });
                tag("div.col-md-4.col-md-offset-4", new Attr("style", "text-align: right"), () -> {
                    writeState(thread.getThreadState());
                });
            });

            tag("div.collapse" + (collapse ? "" : ".in"), new Attr("id", threadId.intValue()), () -> {
                StringBuilder st = new StringBuilder();
                WhatThreadDoes.Handle handle = WhatThreadDoes.byThreadId(thread.getThreadId());
                for (String wtdStackEntry : handle.stack()) {
                    st.append("- ").append(wtdStackEntry).append("\n");
                }
                st.append(ExceptionUtils.printStackTrace(thread));
                preText(st.toString());
            });

            threadId.increment();
        }
    }

    private String hrefWithParam(String name, String value) {
        return UrlUtils.hrefWithParam(name, value, pagePath, params);
    }

    private static EnumMap<Thread.State, String> STATE_COLORS = new EnumMap<>(Thread.State.class);
    static {
        STATE_COLORS.put(Thread.State.NEW, "info");
        STATE_COLORS.put(Thread.State.RUNNABLE, "success");
        STATE_COLORS.put(Thread.State.BLOCKED, "danger");
        STATE_COLORS.put(Thread.State.WAITING, "default");
        STATE_COLORS.put(Thread.State.TIMED_WAITING, "info");
        STATE_COLORS.put(Thread.State.TERMINATED, "warning");
    }

    private void writeState(Thread.State state) {
        tag("span", new Attr("class", "label label-" + STATE_COLORS.get(state)), () -> write(state.name()));
    }
}
