package ru.yandex.solomon.staffOnly.html;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.staffOnly.html.svg.ManagerSvgElement;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public abstract class HtmlWriter {

    public static final Attr[] EMPTY_ATTRS = new Attr[0];

    public interface Callback {
        void run() throws Exception;
    }

    private final Writer underlying;

    public HtmlWriter(Writer underlying) {
        this.underlying = underlying;
    }

    public HtmlWriter() {
        this.underlying = new StringWriter();
    }

    public abstract void run();

    public String genString() {
        run();
        if (!(underlying instanceof StringWriter)) {
            throw new IllegalStateException("wrong writer");
        }
        return underlying.toString();
    }

    public static class Attr {
        @Nonnull
        private final String name;
        @Nonnull
        final String value;

        public Attr(@Nonnull String name, @Nonnull String value) {
            this.value = value;
            this.name = name;

            if (name.isEmpty()) {
                throw new IllegalStateException("name cannot be empty");
            }
        }

        public Attr(@Nonnull String name, int value) {
            this(name, Integer.toString(value));
        }

        @Nonnull
        public String getName() {
            return name;
        }

        @Nonnull
        public String getValue() {
            return value;
        }

        public static Attr cssClass(@Nonnull String value) {
            return new Attr("class", value);
        }

        public static Attr id(@Nonnull String value) {
            return new Attr("id", value);
        }

        @Nonnull
        public static Attr style(String value) {
            return new Attr("style", value);
        }

        @Nonnull
        public static Attr style(CssLine... csss) {
            return style(Arrays.stream(csss).map(CssLine::format).collect(Collectors.joining("; ")));
        }

        public static Attr xmlns(@Nonnull String xmlns) {
            return new Attr("xmlns", xmlns);
        }

        @Override
        public String toString() {
            return name + "=\"" + escapeHtml(value) + "\"";
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            Attr attr = (Attr) o;

            if (!name.equals(attr.name)) return false;
            if (!value.equals(attr.value)) return false;

            return true;
        }

        @Override
        public int hashCode() {
            int result = name.hashCode();
            result = 31 * result + value.hashCode();
            return result;
        }
    }

    public void doctype() {
        line("<!doctype html>");
    }

    public void html(Callback content) {
        blockTag("html", content);
    }

    public void head(Callback content) {
        blockTag("head", content);
    }

    public void title(String title) {
        tag("title", () -> write(title));
    }

    public void scriptHref(String href) {
        tag("script",
                new Attr("language", "javascript"),
                new Attr("type", "text/javascript"),
                new Attr("src", href));
        nl();
    }

    public void linkStylesheet(String href) {
        tag("link", new Attr("rel", "stylesheet"), new Attr("href", href));
        nl();
    }

    public void linkFavicon(String href) {
        tag("link", new Attr("rel", "shortcut icon"), new Attr("href", href));
    }

    public void body(Callback content) {
        blockTag("body", content);
    }

    public void hn(int n, Callback content, Attr... attrs) {
        tag("h" + n, content, attrs);
    }

    public void h1(Callback content, Attr... attrs) {
        hn(1, content, attrs);
    }

    public void h2(Callback content, Attr... attrs) {
        hn(2, content, attrs);
    }

    public void h3(Callback content, Attr... attrs) {
        hn(3, content, attrs);
    }

    public void h4(Callback content, Attr... attrs) {
        hn(4, content, attrs);
    }

    public void h5(Callback content, Attr... attrs) {
        hn(5, content, attrs);
    }

    public void h6(Callback content, Attr... attrs) {
        hn(6, content, attrs);
    }

    public void hn(int n, String content, Attr... attrs) {
        hn(n, () -> write(content), attrs);
    }

    public void h1(String content) {
        hn(1, content);
    }

    public void h2(String content) {
        hn(2, content);
    }

    public void h3(String content) {
        hn(3, content);
    }

    public void h4(String content) {
        hn(4, content);
    }

    public void h5(String content) {
        hn(5, content);
    }

    public void h6(String content) {
        hn(6, content);
    }

    public void div(Callback content, Attr... attrs) {
        blockTag("div", content, attrs);
    }

    public void div(Attr a1, Callback content) {
        div(content, a1);
    }

    public void div(Attr a1, Attr a2, Callback content) {
        div(content, a1, a2);
    }

    public void div(Attr... attrs) {
        div(() -> {}, attrs);
    }

    public void divWithClass(String cssClass, Callback content) {
        div(content, Attr.cssClass(cssClass));
    }

    public void divWithId(String id, Callback content) {
        div(content, Attr.id(id));
    }

    public void b(String text) {
        tag("b", () -> write(text));
    }

    public void b(Number text) {
        tag("b", () -> write(text));
    }

    public void p(String text) {
        tag("p", () -> write(text));
    }

    public void p(Number text) {
        p(text.toString());
    }

    public void table(Callback content) {
        tag("table", content);
    }

    public void tr(Callback content) {
        tag("tr", content);
    }

    public void td(Callback content, Attr... attrs) {
        tag("td", content, attrs);
    }

    public void tdText(String text) {
        td(() -> write(text));
    }

    public void tdText(Number number) {
        td(() -> write(number.toString()), Attr.style("text-align: right"));
    }

    public void tdColspan(int colspan, Callback content) {
        tag("td", new Attr("colspan", String.valueOf(colspan)), content);
    }

    public void th(Callback content) {
        tag("th", content);
    }

    public void thText(String text) {
        th(() -> write(text));
    }

    public void thText(Number text) {
        thText(text.toString());
    }

    public void trThs(String... texts) {
        tr(() -> ths(texts));
    }

    public void trThsNowrap(String... ths) {
        tr(() -> ths(ths), Attr.style("white-space: nowrap"));
    }

    public void tr(Callback callback, Attr... attrs) {
        tag("tr", callback, attrs);
    }

    public void ths(String... texts) {
        for (String text : texts) {
            thText(text);
        }
    }

    public void trThTd(String th, Callback td) {
        tr(() -> {
            thText(th);
            td(td);
        });
    }

    public void trThTd(String th, String td) {
        trThTd(th, () -> write(td));
    }

    public void trThTd(String th, Number td) {
        trThTd(th, () -> write(td));
    }

    public void ul(Callback content, Attr... attrs) {
        tag("ul", content, attrs);
    }

    public void li(Callback content, Attr... attrs) {
        tag("li", content, attrs);
    }

    public void pre(Callback content) {
        tag("pre", content);
    }

    public void preText(String content) {
        pre(() -> write(content));
    }

    public void preWithId(String id, Callback content) {
        tag("pre", Attr.id(id), content);
    }

    public void script(Callback content) {
        blockTag("script", content);
    }

    public void aHref(String href, String title) {
        aHref(href, () -> write(title));
    }

    public void aHref(String href, Attr attr1, String title) {
        aHref(href, attr1, () -> write(title));
    }

    public void aHref(String href, Number title) {
        aHref(href, title.toString());
    }

    public void aHref(String href, Callback content) {
        tag("a", new Attr("href", href), content);
    }

    public void aHref(String href, Attr attr1, Callback content) {
        tag("a", content, new Attr("href", href), attr1);
    }

    public void small(String text) {
        tag("small", () -> write(text));
    }

    public void label(String labelFor, String title) {
        tag("label for=" + labelFor, () -> write(title));
    }

    public void form(Callback content, Attr... attrs) {
        tag("form", content, attrs);
    }

    public void inputHidden(String name, String value) {
        tag("input", new Attr("type", "hidden"), new Attr("name", name), new Attr("value", value));
    }

    public void writeRaw(String s) {
        try {
            underlying.write(s);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static String escapeHtml(String s) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < s.length(); ++i) {
            char c = s.charAt(i);
            if (c == '<') {
                sb.append("&lt");
            } else if (c == '>') {
                sb.append("&gt;");
            } else if (c == '&') {
                sb.append("&amp;");
            } else if (c == '"') {
                sb.append("&quot;");
            } else if (c == '\'') {
                sb.append("&#39;");
            } else {
                sb.append(c);
            }
        }
        return sb.toString();
    }

    public void write(String s) {
        writeRaw(escapeHtml(s));
    }

    public void write(Number s) {
        write(s.toString());
    }

    public void nl() {
        writeRaw("\n");
    }

    public void line(String line) {
        writeRaw(line);
        nl();
    }

    public void tag(String tag, Attr... attrs) {
        open(tag, attrs);
        close(tag);
    }

    public void tag(String tag, Attr attr1, Callback content) {
        open(tag, attr1);
        content(content);
        close(tag);
    }

    public void open(String tag, Attr... attrs) {
        open(TagNameAndAttrs.parse(tag).withMoreAttrs(Arrays.asList(attrs)));
    }

    public void open(TagNameAndAttrs tag) {
        writeRaw("<" + tag.getTagName());
        for (Attr attr : tag.getAttrs()) {
            writeRaw(" " + attr.toString());
        }
        writeRaw(">");
    }

    public void close(String tag) {
        close(TagNameAndAttrs.parse(tag));
    }

    public void close(TagNameAndAttrs tag) {
        writeRaw("</" + tag.getTagName() + ">");
    }

    public void content(Callback callback) {
        try {
            callback.run();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public void blockTag(String tag, Callback content, Attr... attrs) {
        open(tag, attrs);
        nl();
        content(content);
        close(tag);
        nl();
    }

    public void tag(String tag, Callback content, Attr... attrs) {
        open(tag, attrs);
        content(content);
        close(tag);
    }

    public void tag(TagNameAndAttrs tagNameAndAttrs) {
        tag(tagNameAndAttrs.getTagName(), tagNameAndAttrs.getAttrs().toArray(new Attr[0]));
    }


    public void svg(int width, int height, Callback content) {
        tag("svg", content, new Attr("width", width), new Attr("height", height), Attr.xmlns("http://www.w3.org/2000/svg"));
    }

    public void svgElement(ManagerSvgElement svg) {
        tag(svg.toTagNameAndAttrs());
    }
}
