package ru.yandex.solomon.staffOnly.html;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.solomon.util.parser.ParserSupport;

/**
 * @author Stepan Koltsov
 */
public class TagNameAndAttrs {
    private final String tagName;
    private final List<HtmlWriter.Attr> attrs;

    public TagNameAndAttrs(String tagName, List<HtmlWriter.Attr> attrs) {
        this.tagName = tagName;
        this.attrs = attrs;
    }

    public String getTagName() {
        return tagName;
    }

    public List<HtmlWriter.Attr> getAttrs() {
        return attrs;
    }

    public TagNameAndAttrs withMoreAttrs(List<HtmlWriter.Attr> attrs) {
        return new TagNameAndAttrs(tagName,
                Stream.concat(this.attrs.stream(), attrs.stream()).collect(Collectors.toList()));
    }

    private static class Parser extends ParserSupport {

        private static final Pattern OTHER_PATTERN = Pattern.compile("[^ .#]+");
        private static final Pattern EQUALS_PATTERN = Pattern.compile("=");

        public Parser(String string) {
            super(string);
        }

        private String consumeUntilSep() {
            return consume(OTHER_PATTERN);
        }

        public HtmlWriter.Attr parseAttr() {
            if (lookaheadIs("#")) {
                return parseId();
            } else if (lookaheadIs(".")) {
                return parseClass();
            } else if (lookaheadIs(" ")) {
                return parseAny();
            } else {
                throw new RuntimeException("failed to parse attribute");
            }
        }

        private HtmlWriter.Attr parseAny() {
            consume(' ');
            String name = consumeUntil(EQUALS_PATTERN);
            consume('=');
            String value = consumeUntilSep();
            return new HtmlWriter.Attr(name, value);
        }

        private HtmlWriter.Attr parseClass() {
            List<String> classes = new ArrayList<>();
            while (lookaheadIs('.')) {
                consume('.');
                String cssClass = consumeUntilSep();
                classes.add(cssClass);
            }
            return HtmlWriter.Attr.cssClass(classes.stream().collect(Collectors.joining(" ")));
        }

        private HtmlWriter.Attr parseId() {
            consume('#');
            String id = consumeUntilSep();
            return HtmlWriter.Attr.id(id);
        }
    }

    private static List<HtmlWriter.Attr> parseAttrs(String string) {
        Parser parser = new Parser(string);
        return parser.parseSeq(parser::parseAttr);
    }

    private static final Pattern first = Pattern.compile("[#. ]");

    public static TagNameAndAttrs parse(String string) {
        Matcher matcher = first.matcher(string);

        if (matcher.find()) {
            int start = matcher.start();
            String tagName = string.substring(0, start);
            return new TagNameAndAttrs(tagName, parseAttrs(string.substring(start)));
        } else {
            return new TagNameAndAttrs(string, new ArrayList<>());
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        TagNameAndAttrs that = (TagNameAndAttrs) o;

        if (attrs != null ? !attrs.equals(that.attrs) : that.attrs != null) return false;
        if (tagName != null ? !tagName.equals(that.tagName) : that.tagName != null) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = tagName != null ? tagName.hashCode() : 0;
        result = 31 * result + (attrs != null ? attrs.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return tagName + attrs.stream().map(a -> " " + a.toString()).collect(Collectors.joining());
    }
}
