package ru.yandex.solomon.staffOnly.manager;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.openjdk.jol.info.GraphLayout;
import org.springframework.beans.factory.BeanFactoryUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.server.ServerWebExchange;

import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.reflection.MethodX;
import ru.yandex.misc.reflection.TypeX;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.annotations.HideFromManagerUi;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethodArgument;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.html.HtmlWriter.Attr;
import ru.yandex.solomon.staffOnly.html.HtmlWriterWithCommonLibraries;
import ru.yandex.solomon.staffOnly.manager.find.NamedObject;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectFindContext;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.path.ManagerObjectPath;
import ru.yandex.solomon.staffOnly.manager.path.PathElement;

/**
 * @author Stepan Koltsov
 */
@RestController
@Import({
    NamedObjectFindContext.class,
    ManagerWriterContext.class,
})
@ParametersAreNonnullByDefault
public class ManagerController {

    private static final String BEAN_PARAM = "bean";
    private static final String CLASS_ID_PARAM = "class";
    static final String METHOD_PARAM = "method";
    static final String ACTION_PARAM = "action";
    static final String DESCRIPTOR_PARAM = "desc";
    private static final String INVOKE_PARAM = "invoke";
    private static final String FIELD_PARAM = "field";
    static final String LIMIT_PARAM = "limit";

    static final String ACTION_TOTAL_SIZE = "totalSize";
    static final String ACTION_FOOTPRINT = "footprint";

    private static final Attr ATTR_METHOD_POST = new Attr("method", "POST");
    private static final Attr ATTR_METHOD_GET = new Attr("method", "GET");

    @Autowired
    HttpAuthenticator authenticator;
    @Autowired
    InternalAuthorizer authorizer;
    @Autowired
    NamedObjectFindContext namedObjectFindContext;
    @Autowired
    ManagerWriterContext managerWriterContext;

    @Nonnull
    public static String namedObjectLink(NamedObjectId objectId) {
        String url = "/manager";
        url = UrlUtils.addParameter(url, BEAN_PARAM, objectId.getInstanceId());
        if (!objectId.getClassId().equals(NamedObjectFindContext.BEAN_CLASS_ID)) {
            url = UrlUtils.addParameter(url, CLASS_ID_PARAM, objectId.getClassId());
        }
        return url;
    }

    @Nonnull
    public static String namedObjectLink(NamedObject namedObject) {
        return namedObjectLink(namedObject.namedObjectIdGlobal());
    }

    @Nonnull
    public String objectLinkOrThrow(Object bean) {
        NamedObjectId objectId = namedObjectFindContext.findIdForObject(bean)
            .orElseThrow(RuntimeException::new);
        return namedObjectLink(objectId);
    }

    @Nonnull
    public static AHref objectAHref(NamedObject bean) {
        return new AHref(namedObjectLink(bean), bean);
    }

    @Nonnull
    public String objectLink(ManagerObjectPath path) {
        if (path.getFieldPath().isEmpty()) {
            return namedObjectLink(path.getRootId());
        } else {
            return beanFieldLink(path.getRootId(), path.getFieldPathJoined());
        }
    }

    @Nonnull
    public String methodHref(ManagerObjectPath finalPath, Method method) {
        return UrlUtils.addParameter(objectLink(finalPath),
            METHOD_PARAM, method.getName(),
            DESCRIPTOR_PARAM, methodDescriptor(method));
    }

    @Nonnull
    public String actionHref(ManagerObjectPath finalPath, String action) {
        return UrlUtils.addParameter(objectLink(finalPath), ACTION_PARAM, action);
    }

    @Nonnull
    public String beanFieldLink(NamedObjectId objectId, String fieldName) {
        return UrlUtils.addParameter(namedObjectLink(objectId),
            FIELD_PARAM, fieldName);
    }

    @Autowired
    private ApplicationContext applicationContext;

    @Bean
    public RootLink managerLink() {
        return new RootLink("/manager", "Manager");
    }

    @PostMapping(
        path = "/manager",
        consumes = MediaType.APPLICATION_FORM_URLENCODED_VALUE,
        produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> managerPost(
        @RequestParam(value = LIMIT_PARAM, defaultValue = "0") int limit,
        ServerHttpRequest httpRequest,
        ServerWebExchange webExchange)
    {
        return webExchange.getFormData().toFuture()
            .thenCompose(formData -> {
                var postParams = formData.toSingleValueMap();
                return manager(
                    postParams.getOrDefault(BEAN_PARAM, ""),
                    postParams.getOrDefault(CLASS_ID_PARAM, ""),
                    postParams.getOrDefault(METHOD_PARAM, ""),
                    postParams.getOrDefault(ACTION_PARAM, ""),
                    postParams.getOrDefault(DESCRIPTOR_PARAM, ""),
                    Integer.parseInt(postParams.getOrDefault(INVOKE_PARAM, "0")),
                    postParams.getOrDefault(FIELD_PARAM, ""),
                    limit,
                    httpRequest,
                    postParams);
            });
    }

    @GetMapping(path = "/manager", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> managerGet(
        @RequestParam(value = BEAN_PARAM, defaultValue = "") String beanName,
        @RequestParam(value = CLASS_ID_PARAM, defaultValue = "") String classId,
        @RequestParam(value = METHOD_PARAM, defaultValue = "") String methodName,
        @RequestParam(value = ACTION_PARAM, defaultValue = "") String actionName,
        @RequestParam(value = DESCRIPTOR_PARAM, defaultValue = "") String descriptor,
        @RequestParam(value = INVOKE_PARAM, defaultValue = "0") int invoke,
        @RequestParam(value = FIELD_PARAM, defaultValue = "") String field,
        @RequestParam(value = LIMIT_PARAM, defaultValue = "0") int limit,
        ServerHttpRequest httpRequest,
        @RequestParam Map<String, String> queryArgs)
    {
        return manager(
            beanName,
            classId,
            methodName,
            actionName,
            descriptor,
            invoke,
            field,
            limit,
            httpRequest,
            queryArgs);
    }

    private CompletableFuture<String> manager(
        String beanName,
        String classId,
        String methodName,
        String actionName,
        String descriptor,
        int invoke,
        String field,
        int limit,
        ServerHttpRequest httpRequest,
        Map<String, String> methodParams)
    {
        return authenticator.authenticate(httpRequest)
            .thenCompose(authSubject -> authorizer.authorize(authSubject))
            .thenApply(account -> {
                String url = httpRequest.getURI().toString();

                if (beanName.isEmpty()) {
                    return manager();
                } else {
                    return beanPage(classId, beanName, methodName, descriptor, invoke, actionName, field, methodParams, limit, url);
                }
            });
    }

    private Object getObjectByPath(ManagerObjectPath path) {
        Object o = namedObjectFindContext.findObject(path.getRootId());
        for (PathElement fieldName : path.getFieldPath()) {
            o = new ObjectNav(o).lookup(fieldName);
            namedObjectFindContext.checkBeanNotHidden(o);
        }
        return o;
    }

    private String beanPage(
        String classId,
        String beanName,
        String methodName, String descriptor, int invoke,
        String actionName,
        String field,
        Map<String, String> methodParams,
        int limit,
        String url)
    {
        ManagerObjectPath path = ManagerObjectPath.parseBean(classId, beanName, field);

        Object object = getObjectByPath(path);

        if (!methodName.isEmpty()) {
            return objectMethodPage(object, path, methodName, descriptor, invoke, methodParams, url, limit);
        } else if (!actionName.isEmpty()) {
            return objectActionPage(object, actionName);
        } else {
            return new ManagerPageTemplate(path.toString(), ManagerController.this, managerWriterContext) {
                @Override
                protected void content() {
                    objectPage(object, limit, path, url);

                }
            }.genString();
        }
    }

    private String objectActionPage(Object object, String actionName) {
        String result = null;
        switch (actionName) {
            case ACTION_TOTAL_SIZE:
                result = String.valueOf(GraphLayout.parseInstance(object).totalSize());
                break;

            case ACTION_FOOTPRINT:
                result = GraphLayout.parseInstance(object).toFootprint();
                break;
        }

        final String resultFinal = result;
        return new ManagerPageTemplate(StringUtils.capitalize(actionName) + " of " + object.getClass(), this, managerWriterContext) {
            @Override
            protected void content() {
                h3("Result");
                preText(resultFinal);
            }
        }.genString();
    }

    private int getClassInheritanceLevel(@Nullable Class<?> clazz) {
        int result = 0;
        while (clazz != null) {
            result++;
            clazz = clazz.getSuperclass();
        }
        return result;
    }

    private String objectMethodPage(
        Object bean, ManagerObjectPath beanPath, String methodName, String descriptor,
        int invoke, Map<String, String> methodParams, String url, int limit)
    {
        List<Method> matchingMethods = beanMethods(bean)
                .filter(m -> m.getName().equals(methodName))
                .filter(m -> descriptor.isEmpty() || methodDescriptor(m).equals(descriptor))
                .collect(Collectors.toList());

        if (matchingMethods.size() == 0) {
            throw new RuntimeException("No such method found!");
        } else if (matchingMethods.size() > 1) {
            // Calling method from the leaf class
            matchingMethods.sort(Comparator.comparingInt(method -> getClassInheritanceLevel(((Method)method).getDeclaringClass())).reversed());
        }

        Method method = matchingMethods.iterator().next();

        Object r;
        List<String> queryArgsRaw = new ArrayList<>();
        if (invoke != 0) {
            Object[] params = new Object[method.getParameterCount()];
            for (int i = 0; i < method.getParameterCount(); ++i) {
                String queryArgName = "p" + i;
                String s = methodParams.get(queryArgName);
                if (s == null) {
                    throw new RuntimeException("required param not set: " + queryArgName);
                }
                queryArgsRaw.add(s);
                params[i] = ParameterParser.parse(s, TypeX.wrap(method.getGenericParameterTypes()[i]));
            }
            try {
                method.setAccessible(true);
                r = method.invoke(bean, params);

                if (r instanceof CompletableFuture<?>) {
                    r = ((CompletableFuture) r).get(30, TimeUnit.SECONDS);
                }
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        } else {
            r = null;
            for (int i = 0; i < method.getParameterCount(); ++i) {
                queryArgsRaw.add("");
            }
        }

        final Object finalR = r;
        return new ManagerPageTemplate("Method " + methodName, this, managerWriterContext) {
            @Override
            protected void content() {
                buildMethodExecuteButton(queryArgsRaw, beanPath, method, this);

                if (invoke != 0) {
                    h3("Result");

                    objectPage(finalR, limit, null, url);
                }
            }
        }.genString();
    }

    public static void buildMethodExecuteButton(
            List<String> queryArgsRaw, ManagerObjectPath beanPath, Method method, HtmlWriterWithCommonLibraries context) {
        var executeHttpMethod = getExecuteHttpMethod(method);

        context.form(() -> {

            if (!beanPath.getRootId().getClassId().equals(NamedObjectFindContext.BEAN_CLASS_ID)) {
                context.inputHidden(CLASS_ID_PARAM, beanPath.getRootId().getClassId());
            }
            context.inputHidden(BEAN_PARAM, beanPath.getRootId().getInstanceId());
            if (!beanPath.getFieldPath().isEmpty()) {
                context.inputHidden(FIELD_PARAM, beanPath.getFieldPathJoined());
            }
            context.inputHidden(METHOD_PARAM, method.getName());
            context.inputHidden(DESCRIPTOR_PARAM, methodDescriptor(method));
            context.inputHidden(INVOKE_PARAM, "1");
            for (int i = 0; i < method.getParameterCount(); ++i) {
                Parameter parameter = method.getParameters()[i];
                String id = "p" + i;
                String value = queryArgsRaw.get(i);
                context.tag("div.form-group", () -> {
                    context.label("default", getParameterName(parameter) + ": " + parameter.getParameterizedType());
                    context.inputTextfieldFormControl(id, id, value);
                });
            }
            context.buttonSubmitDefault("Execute (" + executeHttpMethod.name() +")");
        }, getExecuteHttpMethodAttr(executeHttpMethod));
    }

    private static ManagerMethod.ExecuteMethod getExecuteHttpMethod(Method method) {
        var annotation = method.getAnnotation(ManagerMethod.class);
        return annotation == null
            ? ManagerMethod.ExecuteMethod.GET
            : annotation.executeMethod();
    }

    private static Attr getExecuteHttpMethodAttr(ManagerMethod.ExecuteMethod executeMethod) {
        return switch (executeMethod) {
            case GET -> ATTR_METHOD_GET;
            case POST -> ATTR_METHOD_POST;
        };
    }

    private static String getParameterName(Parameter parameter) {
        ManagerMethodArgument nameAnnotation = parameter.getAnnotation(ManagerMethodArgument.class);
        if (nameAnnotation == null || nameAnnotation.name().isEmpty()) {
            return parameter.getName();
        } else {
            return nameAnnotation.name();
        }
    }

    public static Stream<Method> beanMethods(@Nonnull Object bean) {
        return ClassX.getClass(bean).getAllDeclaredMethods().stream()
            .filter(m -> !m.isStatic())
            .filter(m -> !m.isAbstract())
            .filter(m -> !m.isBridge())
            .filter(m -> !m.isSynthetic())
            .filter(m -> !m.hasAnnotation(HideFromManagerUi.class))
            .map(MethodX::getMethod);
    }

    static Stream<?> everythingAsStream(Object object) {
        if (object instanceof List<?>) {
            return ((List<?>) object).stream();
        } else if (object.getClass().isArray()) {
            return new ArrayAsList(object).stream();
        } else {
            return Stream.empty();
        }
    }

    private String manager() {
        return new ManagerPageTemplate("Manager", this, managerWriterContext) {
            @Override
            protected void content() {
                tableTable(() -> {
                    String[] names = BeanFactoryUtils.beanNamesIncludingAncestors(applicationContext);
                    Arrays.sort(names);

                    trThs("#", "Bean", "Class");
                    for (int i = 0; i < names.length; i++) {
                        String name = names[i];
                        Object bean = applicationContext.getBean(name);
                        if (bean == null || namedObjectFindContext.isHiddenBean(bean)) {
                            continue;
                        }

                        final int rowNum = i + 1;
                        tr(() -> {
                            tdText(Integer.toString(rowNum));
                            td(() -> {
                                aHref(namedObjectLink(new NamedObjectId(NamedObjectFindContext.BEAN_CLASS_ID, name)), name);
                            });
                            tdText(bean.getClass().getName());
                        });
                    }
                });
            }
        }.genString();
    }

    static String methodDescriptor(Method method) {
        return org.objectweb.asm.commons.Method.getMethod(method).getDescriptor();
    }

}
