package ru.yandex.solomon.staffOnly.manager;

import java.lang.reflect.InaccessibleObjectException;
import java.lang.reflect.Method;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.misc.reflection.Annotated;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.reflection.MethodX;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.path.ManagerObjectPath;
import ru.yandex.solomon.staffOnly.manager.path.PathElement;
import ru.yandex.solomon.staffOnly.manager.special.ExtraContent;
import ru.yandex.solomon.util.ExceptionUtils;

/**
* @author Stepan Koltsov
*/
@ParametersAreNonnullByDefault
abstract class ManagerPageTemplate extends ru.yandex.solomon.staffOnly.www.ManagerPageTemplate {
    private final ManagerController managerController;
    private final ManagerWriterContext managerWriterContext;

    public ManagerPageTemplate(String title, ManagerController managerController, ManagerWriterContext managerWriterContext) {
        super(title);
        this.managerController = managerController;
        this.managerWriterContext = managerWriterContext;
    }

    public interface ExtraContentMethod {
        void invoke(Object thiz, ExtraContentParam p);
    }

    protected void objectPage(@Nullable Object object, int limit, @Nullable ManagerObjectPath path, @Nonnull String url) {
        if (path == null) {
            path = managerWriterContext.namedObjectFindContext.findIdForObject(object)
                .map(n -> new ManagerObjectPath(n, List.of()))
                .orElse(null);
        }

        Optional<ManagerObjectPath> pathO = Optional.ofNullable(path);

        if (object == null) {
            h1("null");
            return;
        }

        if (object instanceof Thread) {
            h1("Stack trace");
            preText(ExceptionUtils.printStackTrace((((Thread) object)).getStackTrace()));
        }

        if (object instanceof Throwable) {
            h1("Stack trace");
            preText(ExceptionUtils.printStackTrace((Throwable) object));
        }

        h1("Fields");
        tableTable(() -> {
            trThs("Property", "Value");

            int j = 0;
            Stream<ObjectNav.PropertyAndValue> properties = new ObjectNav(object).properties();
            int useLimit = limit != 0 ? limit : 1000;
            for (ObjectNav.PropertyAndValue pv : (Iterable<ObjectNav.PropertyAndValue>)properties::iterator) {
                if (isLimit(j, useLimit, url)) {
                    break;
                }

                Optional<ManagerObjectPath> childPath = joinPaths(pathO, pv);
                propertyTr(pv.getName(), pv.getDisplay(), pv.getSource(), childPath.orElse(null));
                ++j;

                if (pv.isPull()) {
                    Stream<ObjectNav.PropertyAndValue> pulledProperites = new ObjectNav(pv.getNavigate()).properties();
                    for (ObjectNav.PropertyAndValue childPv : (Iterable<ObjectNav.PropertyAndValue>)pulledProperites::iterator) {
                        Optional<ManagerObjectPath> childChildPath = joinPaths(childPath, childPv);
                        propertyTr(pv.getName() + "." + childPv.getName(), childPv.getDisplay(), childPv.getSource(), childChildPath.orElse(null));
                        if (isLimit(j, useLimit, url)) {
                            break;
                        }
                        ++j;
                    }

                }
            }
        });

        for (Tuple2<MethodX, ExtraContent> t : ClassX.getClass(object)
            .getAllDeclaredInstanceMethods()
            .filterMap(m -> {
                try {
                    m.setAccessible(true);
                } catch (InaccessibleObjectException | SecurityException e) {
                    return Option.empty();
                }
                return m.getAnnotationO(ExtraContent.class).map(e -> Tuple2.tuple(m, e));
            }))
        {
            MethodX method = t._1;
            ExtraContent extraContent = t._2;
            if (extraContent.value().length() > 0) {
                hn(ExtraContent.H, extraContent.value());
            }

            try {
                method.toSam(ExtraContentMethod.class).invoke(object, new ExtraContentParam(this, managerController.managerWriterContext));
            } catch (Throwable e) {
                preText(ExceptionUtils.printStackTrace(e));
            }
        }


        if (path != null) {
            // not need to show methods if we cannot call them

            List<Method> methods = ManagerController.beanMethods(object)
                .filter(m -> !MethodX.wrap(m).isStatic())
                .filter(m -> m.getDeclaringClass() != Object.class)
                .collect(Collectors.groupingBy(method -> method.getName() + "^" + ManagerController.methodDescriptor(method)))
                .values().stream()
                .map(list -> list.get(0))
                .sorted(Comparator.comparing(Method::getName))
                .collect(Collectors.toList());

            List<Method> managerMethods = methods.stream()
                    .filter(m -> m.isAnnotationPresent(ManagerMethod.class))
                    .collect(Collectors.toList());

            if (!managerMethods.isEmpty()) {
                h1("Manager-oriented methods");
                renderMethods(path, managerMethods);
            }

            h1("Methods");
            renderMethods(path, methods);

            h1("Actions");
            divWithClass("alert alert-warning", () -> {
                b("CAUTION!");
                p("Please do not call this actions for huge objects, " +
                    "because they can be very slow and even crash " +
                    "current process (e.g. in case of memory overflow).");
            });
            renderActions(path, ManagerController.ACTION_TOTAL_SIZE, ManagerController.ACTION_FOOTPRINT);
        }
    }

    private Optional<ManagerObjectPath> joinPaths(Optional<ManagerObjectPath> parentPath, ObjectNav.PropertyAndValue pv) {
        if (parentPath.isPresent()) {
            Optional<PathElement> elementPath = pv.getPathElementO();
            if (elementPath.isPresent()) {
                return Optional.of(parentPath.get().child(elementPath.get()));
            }
        }
        return Optional.empty();
    }

    private boolean isLimit(int j, int useLimit, String url) {
        if (j == useLimit) {
            tr(() -> {
                tdColspan(2, () -> {
                    write("... ");
                    int nextLimit = useLimit * 10;
                    String moreHref = updateParameter(url, ManagerController.LIMIT_PARAM, nextLimit);
                    aHref(moreHref, "show first " + nextLimit);
                });
            });
            return true;
        }
        return false;
    }

    private void renderActions(ManagerObjectPath finalPath, String... actions) {
        tableTable(() -> {
            trThs("Action");
            for (String action : actions) {
                tr(() -> {
                    td(() -> {
                        aHref(managerController.actionHref(finalPath, action), action);
                    });
                });
            }
        });
    }

    private void renderMethods(ManagerObjectPath finalPath, List<Method> methods) {
        tableTable(() -> {
            trThs("Method");
            methods.forEach(method -> {
                tr(() -> {
                    td(() -> {
                        aHref(managerController.methodHref(finalPath, method),
                            method.getName() + " " + ManagerController.methodDescriptor(method));
                    });
                });
            });
        });
    }

    static String updateParameter(String url, String parameter, int value) {
        int start = url.indexOf(parameter + "=");
        if (start != -1 && (start == 0 || url.charAt(start - 1) == '?' || url.charAt(start - 1) == '&')) {
            int valueStart = start + parameter.length() + 1;
            int valueEnd = url.indexOf('&', valueStart);
            String suffix;
            if (valueEnd == -1) {
                suffix = "";
            } else {
                suffix = url.substring(valueEnd);
            }
            return url.substring(0, valueStart) + value + suffix;
        } else {
            String separator = (url.indexOf('?') == -1) ? "?" : "&";
            return url + separator + parameter + "=" + value;
        }
    }

    private void propertyTr(@Nonnull String name, @Nullable Object value, @Nullable Annotated source, @Nullable ManagerObjectPath childPath) {

        Optional<NamedObjectId> fieldBeanName = managerWriterContext.namedObjectFindContext.findIdForObject(value);
        tr(() -> {
            th(() -> {
                if (fieldBeanName.isPresent()) {
                    aHref(ManagerController.namedObjectLink(fieldBeanName.get()), name);
                } else {
                    if (childPath != null) {
                        aHref(managerController.objectLink(childPath), name);
                    } else {
                        write(name);
                    }
                }
            });
            td(() -> {
                ManagerWriter managerWriter = new ManagerWriter(managerWriterContext, this);
                managerWriter.writeCellValue(value, source);
            });
        });
    }
}
