package ru.yandex.solomon.staffOnly.manager;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Message;

import ru.yandex.misc.reflection.Annotated;
import ru.yandex.misc.reflection.FieldX;
import ru.yandex.solomon.staffOnly.html.CssLine;
import ru.yandex.solomon.staffOnly.html.HtmlAttrs;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.html.HtmlWriterWithCommonLibraries;
import ru.yandex.solomon.staffOnly.manager.special.DisplayAsHex;
import ru.yandex.solomon.staffOnly.manager.special.DisplayAsHexImpl;
import ru.yandex.solomon.staffOnly.manager.special.DurationMillis;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;
import ru.yandex.solomon.staffOnly.manager.table.HtmlWriterTableDef;
import ru.yandex.solomon.staffOnly.manager.table.TableColumn;
import ru.yandex.solomon.staffOnly.manager.table.TableColumnDef;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.solomon.util.collection.array.ArrayBuilder;
import ru.yandex.solomon.util.protobuf.ProtobufText;
import ru.yandex.solomon.util.reflect.reified.Reified;
import ru.yandex.solomon.util.reflect.reified.ReifiedToString;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Stepan Koltsov
 */
public class ManagerWriter {

    private final ManagerWriterContext context;
    private final HtmlWriterWithCommonLibraries htmlWriter;

    @ParametersAreNonnullByDefault
    public ManagerWriter(ManagerWriterContext context, HtmlWriterWithCommonLibraries htmlWriter) {
        this.context = context;
        this.htmlWriter = htmlWriter;
    }

    public ManagerWriterContext getContext() {
        return context;
    }

    public HtmlWriterWithCommonLibraries getHtmlWriter() {
        return htmlWriter;
    }

    public static class AlignedRight {
        @Nullable
        private final Object value;

        public AlignedRight(@Nullable Object value) {
            this.value = value;
        }

        @Override
        public String toString() {
            return Objects.toString(value);
        }
    }

    public void writeCellValue(@Nullable Object value, @Nullable Annotated annotated) {
        var effectiveValue = maybePrettifyValue(value, annotated);

        TableColumn annotation = annotated != null ? annotated.getAnnotation(TableColumn.class) : null;
        if (annotation != null && annotation.pre()) {
            htmlWriter.pre(() -> writeCellValue(effectiveValue));
        } else {
            writeCellValue(effectiveValue);
        }
    }

    @Nullable
    private static Object maybePrettifyValue(@Nullable Object value, @Nullable Annotated annotated) {
        if (value instanceof Long longValue && annotated != null) {
            if (longValue > 0 && annotated.hasAnnotation(InstantMillis.class)) {
                return InstantUtils.formatToMillis(longValue);
            }

            if (longValue >= 0 && annotated.hasAnnotation(DurationMillis.class)) {
                return DurationUtils.millisToSecondsString(longValue) + "s";
            }
        }

        Optional<String> v;
        if (annotated != null && annotated.hasAnnotation(DisplayAsHex.class) &&
            (v = DisplayAsHexImpl.display(value)).isPresent()) {
            return v.get();
        }

        if (value != null &&
            annotated instanceof FieldX field && field.getType().isAssignableTo(ReifiedToString.class)) {
            try {
                Class<?> typeParameter = field.getGenericType().getActualTypeArguments().single().erasure().getClazz();
                return new Reified((ReifiedToString) value, typeParameter);
            } catch (Exception ignore) {
            }
        }

        return value;
    }

    public void writeCellValue(@Nullable Object value) {
        if (value instanceof AlignedRight) {
            writeCellValue(((AlignedRight) value).value);
        } else if (value instanceof Throwable) {
            htmlWriter.preText(ExceptionUtils.printStackTrace((Throwable) value));
        } else if (value == null) {
            htmlWriter.write("null");
        } else if (value instanceof Message) {
            htmlWriter.preText(ProtobufText.serializeToText((Message) value));
        } else if (value instanceof String) {
            String string = (String) value;
            if (string.length() > 200) {
                htmlWriter.write(string.substring(0, 100) + "..." + string.substring(string.length() - 100));
            } else {
                htmlWriter.write(string);
            }
        } else if (value instanceof byte[]) {
            writeCellValue(Arrays.toString((byte[]) value));
        } else if (value instanceof short[]) {
            writeCellValue(Arrays.toString((short[]) value));
        } else if (value instanceof int[]) {
            writeCellValue(Arrays.toString((int[]) value));
        } else if (value instanceof long[]) {
            writeCellValue(Arrays.toString((long[]) value));
        } else if (value instanceof boolean[]) {
            writeCellValue(Arrays.toString((boolean[]) value));
        } else if (value instanceof float[]) {
            writeCellValue(Arrays.toString((float[]) value));
        } else if (value instanceof double[]) {
            writeCellValue(Arrays.toString((double[]) value));
        } else if (value instanceof Object[]) {
            writeCellValue(Arrays.toString((Object[]) value));
        } else if (value instanceof WritableToHtml) {
            ((WritableToHtml) value).writeTo(this);
        } else if (value instanceof Collection<?>) {
            htmlWriter.write(CollectionToString.collectionToString((Collection<?>) value, 200));
        } else if (value instanceof Map<?, ?>) {
            htmlWriter.write(CollectionToString.mapToString((Map<?, ?>) value, 200));
        } else {
            try {
                writeCellValue(value.toString());
            } catch (Throwable t) {
                writeCellValue(t);
            }
        }
    }

    private void writeCellTdWithAttrs(
        @Nullable Object value, @Nullable Annotated annotated, List<HtmlWriter.Attr> attrs)
    {
        boolean alignRight =
            value instanceof Number || value instanceof AlignedRight
                || (annotated != null && annotated.hasAnnotation(DisplayAsHex.class));

        boolean monospaced =
            annotated != null && annotated.hasAnnotation(DisplayAsHex.class);

        ArrayBuilder<CssLine> csss = new ArrayBuilder<>(CssLine.class);
        if (alignRight) {
            csss.add(new CssLine("text-align", "right"));
        }
        if (monospaced) {
            csss.add(CssLine.fontFamilyMonospace());
        }

        HtmlAttrs attrMap = new HtmlAttrs(attrs);

        if (!csss.isEmpty()) {
            attrMap.setDefault(HtmlWriter.Attr.style(csss.build()));
        }

        htmlWriter.td(() -> {
            writeCellValue(value, annotated);
        }, attrMap.buildList().toArray(new HtmlWriter.Attr[0]));
    }

    public void writeCellTd(@Nullable Object value, @Nullable Annotated annotated) {
        writeCellTdWithAttrs(value, annotated, List.of());
    }

    public void writeCellTdWithAttrs(@Nullable Object value, List<HtmlWriter.Attr> tdAttrs) {
        writeCellTdWithAttrs(value, null, tdAttrs);
    }

    public void writeCellTd(@Nullable Object value) {
        writeCellTd(value, null);
    }

    public <K, V> void mapAsTable(Map<K, V> map) {
        htmlWriter.tableTable(() -> {
            for (Map.Entry<K, V> e : map.entrySet()) {
                htmlWriter.tr(() -> {
                    htmlWriter.th(() -> {
                        writeCellValue(e.getKey());
                    });
                    writeCellTd(e.getValue());
                });
            }
        });
    }

    public <T> void reflectVericalTable(T t) {
        reflectVericalTable(t, () -> {});
    }

    public <T> void reflectVericalTable(T t, Runnable extraRows) {
        htmlWriter.tableTable(() -> {
            reflectTrsThTd(t);
            extraRows.run();
        });
    }

    public <T> void reflectTrsThTd(T t) {
        HtmlWriterTableDef<T> tableDef = new HtmlWriterTableDef<>((Class<T>) t.getClass());
        for (TableColumnDef<T> column : tableDef.getColumns()) {
            htmlWriter.tr(() -> {
                column.writeTitleTh(this);
                column.writeValueTd(t, this);
            });
        }
    }

    private <T> void listTableImpl(Collection<T> rows, HtmlWriterTableDef<T> tableDef) {
        htmlWriter.tableTable(() -> {
            htmlWriter.tr(() -> {
                for (TableColumnDef<T> column : tableDef.getColumns()) {
                    column.writeTitleTh(this);
                }
            });
            for (T row : rows) {
                HtmlWriter.Attr[] attrs = (row instanceof WithAttrs)
                    ? ((WithAttrs) row).attrs()
                    : HtmlWriter.EMPTY_ATTRS;

                htmlWriter.tr(() -> {
                    for (TableColumnDef<T> column : tableDef.getColumns()) {
                        column.writeValueTd(row, this);
                    }
                }, attrs);
            }
        });
    }

    public final <T> void listTable(Collection<T> rows, List<TableColumnDef<T>> colums) {
        listTableImpl(rows, new HtmlWriterTableDef<>(colums));
    }

    @SafeVarargs
    public final <T> void listTable(Collection<T> rows, TableColumnDef<T>... columns) {
        listTable(rows, Arrays.asList(columns));
    }

    @SafeVarargs
    public final <T> void reflectListTable(List<T> rows, Class<T> clazz, TableColumnDef<T>... columns) {
        listTableImpl(rows, new HtmlWriterTableDef<>(clazz, columns));
    }

}
