package ru.yandex.solomon.staffOnly.manager;

import java.lang.reflect.Array;
import java.lang.reflect.InaccessibleObjectException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.bolts.collection.Try;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.reflection.Annotated;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.solomon.staffOnly.annotations.HideFromManagerUi;
import ru.yandex.solomon.staffOnly.manager.path.PathElement;
import ru.yandex.solomon.staffOnly.manager.special.PullHere;
import ru.yandex.solomon.staffOnly.manager.table.HtmlWriterReflectColumnDef;

/**
 * @author Stepan Koltsov
 */
public class ObjectNav {

    @Nullable
    private final Object object;

    public ObjectNav(@Nullable Object object) {
        this.object = object;
    }

    public Stream<PropertyAndValue> properties() {
        List<Stream<PropertyAndValue>> r = new ArrayList<>();
        if (object == null) {
            return Stream.empty();
        }

        r.add(Stream.of(
            new PropertyAndValue("Type", object.getClass().getName()),
            new PropertyAndValue("toString()", object))
        );

        if (object.getClass().isArray()) {
            r.add(Stream.of(new PropertyAndValue("length", Array.getLength(object))));
        } else if (object instanceof Collection<?>) {
            r.add(Stream.of(new PropertyAndValue("size()", ((Collection<?>) object).size())));
        } else if (object instanceof Map<?, ?>) {
            r.add(Stream.of(new PropertyAndValue("size()", ((Map<?, ?>) object).size())));
        }

        List<? extends HtmlWriterReflectColumnDef<Object>> columns = HtmlWriterReflectColumnDef
                .tableColumnsFromMethods((Class<Object>) object.getClass());

        r.add(columns.stream().map(columnDef -> new PropertyAndValue(columnDef.title(), columnDef.bind(object))));

        if (object instanceof byte[]) {
            r.add(Stream.of(new PropertyAndValue("hex", Hex.encodeHr((byte[]) object))));
        }

        r.add(ClassX.getClass(object).getAllDeclaredInstanceFields().stream()
            .filter(field -> !field.hasAnnotation(HideFromManagerUi.class))
            .map(field -> {
            Object objectOrThrowable;
            try {
                objectOrThrowable = field.getAccessible(object);
            } catch (Throwable t) {
                objectOrThrowable = t;
            }

            boolean pull = field.hasAnnotation(PullHere.class);
            PathElement pathElement = PathElement.field(field.getName());
            return new PropertyAndValue(field.getName(), pathElement, objectOrThrowable, field, objectOrThrowable, pull);
        }));

        r.add(ClassX.getClass(object).getAllDeclaredInstanceMethods().stream().flatMap(method -> {
            try {
                method.setAccessible(true);
            } catch (InaccessibleObjectException | SecurityException e) {
                return Stream.empty();
            }

            if (!method.hasAnnotation(PullHere.class)) {
                return Stream.empty();
            }

            Object value = Try.tryCatchThrowable(() -> method.invoke(object)).toEither().fold(x -> x, x -> x);

            return Stream.of(new PropertyAndValue(method.getName(), null, value, method, value, !(value instanceof Throwable)));
        }));

        AtomicInteger elementIdx = new AtomicInteger();
        r.add(ManagerController.everythingAsStream(object).map(element -> {
                int i = elementIdx.getAndIncrement();
                return new PropertyAndValue(
                    "[" + i + "]",
                    PathElement.index(i),
                    element,
                    null,
                    element,
                    false);
            }));

        if (object instanceof Map<?, ?>) {
            r.add(((Map<?, ?>) object).entrySet().stream()
                .map(entry -> {
                    PathElement pathElement;
                    if (entry.getKey() instanceof String || entry.getKey() instanceof Number) {
                        // TODO: only if string is safe
                        pathElement = PathElement.mapKey(entry.getKey().toString());
                    } else {
                        pathElement = null;
                    }
                    String title;
                    if (entry.getKey() instanceof String) {
                        title = "'" + entry.getKey() + "'";
                    } else {
                        title = entry.getKey().toString();
                    }
                    return new PropertyAndValue(
                        "[" + title + "]",
                        pathElement,
                        entry.getValue(),
                        null,
                        entry.getValue(),
                        false);
                })
            );
        }

        // this is a flatMap substitution since it cannot into laziness https://bugs.openjdk.java.net/browse/JDK-8075939
        return r.stream().reduce(Stream.empty(), Stream::concat);
    }

    public Object lookup(PathElement pathElement) {
        return properties()
                .filter(pv -> pv.pathElement != null && pv.pathElement.equals(pathElement))
                .map(PropertyAndValue::getNavigate)
                .map(o -> Objects.requireNonNull(o, "property is null: " + pathElement))
                .findAny()
                .get();
    }

    public static class PropertyAndValue {
        @Nonnull
        private final String name;
        @Nullable
        private final PathElement pathElement;
        private final Object display;
        @Nullable
        private final Annotated source;
        private final Object navigate;
        private final boolean pull;

        public PropertyAndValue(@Nonnull String name, @Nullable PathElement pathElement,
            Object display, @Nullable Annotated source, Object navigate, boolean pull)
        {
            this.name = name;
            this.pathElement = pathElement;
            this.display = display;
            this.source = source;
            this.navigate = navigate;
            this.pull = pull;
        }

        public PropertyAndValue(@Nonnull String name, Object display) {
            this.name = name;
            this.pull = false;
            this.pathElement = null;
            this.display = display;
            this.source = null;
            this.navigate = null;
        }

        @Nonnull
        public String getName() {
            return name;
        }

        @Nullable
        public PathElement getPathElement() {
            return pathElement;
        }

        public Optional<PathElement> getPathElementO() {
            return Optional.ofNullable(pathElement);
        }

        public Object getNavigate() {
            return navigate;
        }

        @Nullable
        public Annotated getSource() {
            return source;
        }

        @Nullable
        public Object getDisplay() {
            return display;
        }

        public boolean isPull() {
            return pull;
        }
    }
}
