package ru.yandex.solomon.staffOnly.manager.find;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import org.springframework.beans.factory.BeanFactoryUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.reflection.MemberX;
import ru.yandex.misc.reflection.MethodX;
import ru.yandex.misc.reflection.TypeX;
import ru.yandex.solomon.staffOnly.annotations.HideFromManagerUi;
import ru.yandex.solomon.staffOnly.manager.ParameterParser;
import ru.yandex.solomon.staffOnly.manager.find.annotation.NamedObjectFinderAnnotation;

/**
 * @author Stepan Koltsov
 */
@Component
public class NamedObjectFindContext implements InitializingBean {

    public static final String BEAN_CLASS_ID = "bean";

    @Autowired
    private ApplicationContext applicationContext;
    @Autowired(required = false)
    private NamedObjectFinder[] namedObjectFinders;

    private HashMap<String, NamedObjectFinder> finderByClass;

    private void registerFinder(@Nonnull NamedObjectFinder finder) {
        NamedObjectFinder old = finderByClass.put(finder.classId(), finder);
        if (old != null) {
            throw new IllegalStateException("non-unique finder for class: " + finder.classId());
        }
    }

    private void registerFindersFromBean(@Nonnull Object bean) {
        for (MethodX method : ClassX.wrap(bean.getClass()).getAllDeclaredMethods()
            .filter(MemberX::isNotStatic))
        {
            NamedObjectFinderAnnotation annotation = method.getAnnotation(NamedObjectFinderAnnotation.class);
            if (annotation == null) {
                continue;
            }

            Preconditions.checkArgument(method.getParameters().size() == 1, "wrong parameters count %s", method);
            TypeX type = method.getParameters().get(0).getType();

            Function<String, Object> parameterParser = ParameterParser.createParser(type);
            Preconditions.checkNotNull(parameterParser, "cannot parse parameter of method %s", method);

            String classId = method.getReturnType().getName();
            method.setAccessible(true);

            registerFinder(new NamedObjectFinder() {
                @Nonnull
                @Override
                public String classId() {
                    return classId;
                }

                @Nonnull
                @Override
                public Object find(@Nonnull String id) {
                    return method.invoke(bean, parameterParser.apply(id));
                }
            });
        }
    }

    @Override
    public void afterPropertiesSet() {
        finderByClass = new HashMap<>();

        if (namedObjectFinders != null) {
            for (NamedObjectFinder objectFinder : namedObjectFinders) {
                registerFinder(objectFinder);
            }
        }

        for (Object bean : BeanFactoryUtils.beansOfTypeIncludingAncestors(applicationContext, null, false, false).values()) {
            if (bean != null && !isHiddenBean(bean)) {
                registerFindersFromBean(bean);
            }
        }

        registerFinder(new NamedObjectFinder() {
            @Nonnull
            @Override
            public String classId() {
                return BEAN_CLASS_ID;
            }

            @Nonnull
            @Override
            public Object find(@Nonnull String id) {
                return checkBeanNotHidden(applicationContext.getBean(id));
            }
        });
    }

    public Object checkBeanNotHidden(Object bean) {
        if (bean != null && isHiddenBean(bean)) {
            throw new IllegalStateException("no way to get hidden bean: " + bean.getClass());
        }
        return bean;
    }

    public boolean isHiddenBean(Object bean) {
        String packageName = bean.getClass().getPackageName();
        return bean.getClass().getAnnotation(HideFromManagerUi.class) != null
            || packageName.startsWith("org.spring")
            || packageName.startsWith("org.apache.catalina");
    }

    @Nonnull
    public Object findObject(@Nonnull NamedObjectId objectId) {
        NamedObjectFinder finder = finderByClass.get(objectId.getClassId());
        if (finder == null) {
            throw new IllegalArgumentException("unknown classId: " + objectId.getClassId());
        }
        return checkBeanNotHidden(finder.find(objectId.getInstanceId()));
    }

    public Optional<NamedObjectId> findIdForObject(@Nullable Object bean) {
        if (bean == null || isHiddenBean(bean)) {
            return Optional.empty();
        }

        if (bean instanceof NamedObject) {
            return Optional.of(((NamedObject) bean).namedObjectIdGlobal());
        }

        for (Map.Entry<String, ?> e : BeanFactoryUtils.beansOfTypeIncludingAncestors(applicationContext, bean.getClass()).entrySet()) {
            if (e.getValue() == bean) {
                return Optional.of(new NamedObjectId(BEAN_CLASS_ID, e.getKey()));
            }
        }
        return Optional.empty();
    }
}
