package ru.yandex.solomon.staffOnly.manager.flamegraph;

import java.time.Instant;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class FlameGraphProducerImpl implements FlameGraphProducer, AutoCloseable {
    private final ScheduledExecutorService timer;
    private final ExecutorService executor;
    private volatile ScheduledFuture<?> scheduledFuture = null;
    private ThreadInspector threadInspector = null;
    @SuppressWarnings("FieldCanBeLocal")
    private ActorRunner actor = null;
    private Trie.ImmutableTrieNode lastFlameGraph = null;
    private Instant lastFlameGraphTaken = Instant.EPOCH;
    private StateChanged stateChanged = new StateChanged(State.EMPTY, Instant.EPOCH);
    private double dropFramesPercent = 0.5;
    private int depth = 30;
    private String threadNameRegex = ".*";
    private int samplingFrequency = 100;

    public FlameGraphProducerImpl(@WillNotClose ScheduledExecutorService timer) {
        this.timer = timer;
        this.executor = Executors.newSingleThreadExecutor();
    }

    @Override
    synchronized public void setThreadNameRegex(String regex) {
        this.threadNameRegex = regex;
    }

    @Override
    synchronized public String getThreadNameRegex() {
        return threadNameRegex;
    }

    @Override
    synchronized public void setMaxStackDepth(int depth) {
        this.depth = depth;
    }

    @Override
    synchronized public int getMaxStackDepth() {
        return depth;
    }

    @Override
    synchronized public void setDropFramesPercent(double dropFramesPercent) {
        this.dropFramesPercent = dropFramesPercent;
    }

    @Override
    synchronized public double getDropFramesPercent() {
        return dropFramesPercent;
    }

    @Override
    synchronized public void setSamplingFrequency(int samplingFrequency) {
        this.samplingFrequency = samplingFrequency;
    }

    @Override
    synchronized public int getSamplingFrequency() {
        return samplingFrequency;
    }

    @Override
    synchronized public void start() {
        if (scheduledFuture != null) {
            return;
        }
        var pattern = Pattern.compile(getThreadNameRegex());
        threadInspector = new ThreadInspector(x -> pattern.matcher(x).matches(), getMaxStackDepth());
        actor = new ActorRunner(threadInspector::wake, executor);
        long intervalMicros = (long) Math.max(1000d, 1000_000d / samplingFrequency);
        scheduledFuture = timer.scheduleAtFixedRate(actor::schedule, 0, intervalMicros, TimeUnit.MICROSECONDS);
        stateChanged = new StateChanged(State.RUNNING, Instant.now());
    }

    @Override
    synchronized public void stop() {
        if (scheduledFuture == null) {
            return;
        }
        scheduledFuture.cancel(true);
        scheduledFuture = null;
        doRefreshFlameGraph();
        threadInspector = null;
        stateChanged = new StateChanged(State.DONE, Instant.now());
    }

    @Override
    synchronized public void refresh() {
        if (threadInspector != null) {
            doRefreshFlameGraph();
        }
    }

    private void doRefreshFlameGraph() {
        lastFlameGraph = threadInspector.getRoot();
        lastFlameGraphTaken = Instant.now();
    }

    @Override
    synchronized public FlameGraphWithTime get() {
        var flameGraph = Nullables.orDefault(lastFlameGraph, Trie.ImmutableTrieNode.EMPTY);
        return new FlameGraphWithTime(flameGraph, lastFlameGraphTaken);
    }

    @Override
    public void close() throws Exception {
        var future = scheduledFuture;
        if (future != null) {
            future.cancel(true);
        }
        executor.shutdown();
    }

    @Override
    synchronized public StateChanged getStateChanged() {
        return stateChanged;
    }
}
