package ru.yandex.solomon.staffOnly.manager.flamegraph;

import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;
import java.util.Arrays;
import java.util.Iterator;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class ThreadInspector {

    private final ThreadMXBean threadMXBean = ManagementFactory.getThreadMXBean();
    private final Predicate<String> nameMatcher;
    private final int depth;
    private final Trie trie = new Trie();

    public ThreadInspector(Predicate<String> nameMatcher, int depth) {
        this.nameMatcher = nameMatcher;
        this.depth = depth;
    }

    private static class ReverseArrayMappingIterator<T, U> implements Iterator<U> {
        private int pos;
        private final T[] array;
        private final Function<T, U> transform;
        private final int lb;


        public ReverseArrayMappingIterator(T[] array, int limit, Function<T, U> transform) {
            this.pos = array.length - 1;
            this.array = array;
            this.transform = transform;
            this.lb = Math.max(0, array.length - limit);
        }

        @Override
        public boolean hasNext() {
            return pos >= lb;
        }

        @Override
        public U next() {
            return transform.apply(array[pos--]);
        }
    }

    public void wake() {
        Arrays.stream(threadMXBean.dumpAllThreads(false, false))
                .filter(tinfo -> tinfo.getThreadState() == Thread.State.RUNNABLE && nameMatcher.test(tinfo.getThreadName()))
                .forEach(tinfo -> trie.insert(new ReverseArrayMappingIterator<>(
                        tinfo.getStackTrace(),
                        depth,
                        ThreadInspector::formatTraceElement
                )));
    }

    private static String formatTraceElement(StackTraceElement element) {
        String fileName = element.getFileName();
        int lineNumber = element.getLineNumber();

        return element.getClassName() + "." + element.getMethodName() + "(" +
                (element.isNativeMethod() ? "Native Method)" :
                        (fileName != null && lineNumber >= 0 ?
                                fileName + ":" + lineNumber + ")" :
                                (fileName != null ? fileName + ")" : "Unknown Source)")));
    }

    public Trie.ImmutableTrieNode getRoot() {
        return trie.snapshot();
    }
}
