package ru.yandex.solomon.staffOnly.manager.path;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * @author Stepan Koltsov
 */
public class PathElement {

    private static final Pattern INDEX_PATTERN = Pattern.compile("\\[(\\d+)\\]");

    @Nullable
    private final String field;
    private final int index;

    private PathElement(@Nullable String field, int index) {
        this.field = field;
        this.index = index;
    }

    public static PathElement field(@Nonnull String name) {
        return new PathElement(name, -1);
    }

    public static PathElement index(int index) {
        return new PathElement(null, index);
    }

    public static PathElement mapKey(@Nonnull String key) {
        return field(key);
    }

    public static PathElement parse(String string) {
        Matcher indexMatcher = INDEX_PATTERN.matcher(string);
        if (indexMatcher.matches()) {
            return index(Integer.parseInt(indexMatcher.group(1)));
        }
        return field(string);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        PathElement that = (PathElement) o;

        if (index != that.index) return false;
        if (field != null ? !field.equals(that.field) : that.field != null) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = field != null ? field.hashCode() : 0;
        result = 31 * result + index;
        return result;
    }

    public interface Dispatch<A> {
        A fieldOrMapKey(@Nonnull String field);
        A index(int index);
    }

    public <A> A dispatch(Dispatch<A> dispatch) {
        if (field != null) {
            return dispatch.fieldOrMapKey(field);
        } else {
            return dispatch.index(index);
        }
    }

    @Override
    public String toString() {
        return dispatch(new Dispatch<String>() {
            @Override
            public String fieldOrMapKey(@Nonnull String field) {
                return field;
            }

            @Override
            public String index(int index) {
                return "[" + index + "]";
            }
        });
    }
}
