package ru.yandex.solomon.staffOnly.manager.table;

import java.lang.reflect.Field;
import java.lang.reflect.InaccessibleObjectException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.reflection.Annotated;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.reflection.FieldX;
import ru.yandex.misc.reflection.MethodX;
import ru.yandex.solomon.staffOnly.manager.ManagerWriter;

/**
* @author Stepan Koltsov
*/
public class HtmlWriterReflectColumnDef<T> implements TableColumnDef<T> {
    private final Function<Object, Object> getter;
    private final Class<?> getterResultType;
    private final String nameFieldGetter;
    private final Annotated annotated;

    private HtmlWriterReflectColumnDef(Method getter) {
        MethodX wrap = MethodX.wrap(getter);
        wrap.setAccessible(true);

        this.getter = o -> wrap.invoke(o);
        this.getterResultType = wrap.getReturnType0();
        this.nameFieldGetter = getter.getName().replaceFirst("^get", "");
        annotated = wrap;
    }

    private HtmlWriterReflectColumnDef(Field getter) {
        FieldX wrap = FieldX.wrap(getter);
        wrap.setAccessible(true);

        this.getter = wrap::get;
        this.getterResultType = getter.getType();
        this.nameFieldGetter = getter.getName();
        annotated = wrap;
    }

    public static <T> List<HtmlWriterReflectColumnDef<T>> tableColumns(Class<T> clazz) {
        Stream<HtmlWriterReflectColumnDef<T>> methods = tableColumnsFromMethods(clazz).stream();
        Stream<HtmlWriterReflectColumnDef<T>> fields = ClassX.wrap(clazz).getAllDeclaredInstanceFields().stream()
                .filter(f -> f.getField().getAnnotation(TableColumn.class) != null)
                .map(f -> new HtmlWriterReflectColumnDef<>(f.getField()));
        List<HtmlWriterReflectColumnDef<T>> r = Stream.concat(fields, methods).collect(Collectors.toList());
        if (!r.isEmpty()) {
            return r;
        } else {
            return ClassX.wrap(clazz).getAllDeclaredInstanceFields()
                .filterMap(f -> {
                    if (f.isSynthetic()) {
                        return Option.empty();
                    }
                    try {
                        f.setAccessible(true);
                    } catch (InaccessibleObjectException | SecurityException e) {
                        return Option.empty();
                    }
                    return Option.of(new HtmlWriterReflectColumnDef<>(f.getField()));
                });
        }
    }

    public static <T> List<HtmlWriterReflectColumnDef<T>> tableColumnsFromMethods(Class<T> clazz) {
        return ClassX.wrap(clazz).getAllDeclaredInstanceMethods()
            .filterMap(m -> {
                try {
                    m.setAccessible(true);
                } catch (InaccessibleObjectException | SecurityException e) {
                    return Option.empty();
                }
                return Option.of(m);
            })
            .filter(m -> m.hasAnnotation(TableColumn.class))
            .map(MethodX::getMethod)
            .filter(m -> m.getParameterTypes().length == 0)
            .map(m -> new HtmlWriterReflectColumnDef<T>(m));
    }

    @Nullable
    private TableColumn tableColumn() {
        return annotated.getAnnotation(TableColumn.class);
    }

    @Override
    public void writeValueTd(T row, ManagerWriter writer) {
        Object value;
        try {
            value = getter.apply(row);
        } catch (Throwable e) {
            writer.writeCellValue(e);
            return;
        }

        writer.writeCellTd(value, annotated);
    }

    @Override
    public void writeValue(T row, ManagerWriter writer) {
        Object value;
        try {
            value = getter.apply(row);
        } catch (Throwable e) {
            writer.writeCellValue(e);
            return;
        }

        writer.writeCellValue(value, annotated);
    }

    @Override
    public String title() {
        String title;
        TableColumn annotation = tableColumn();
        if (annotation == null || annotation.value().isEmpty()) {
            title = this.nameFieldGetter.replaceFirst("^get", "");
        } else {
            title = annotation.value();
        }
        return title;
    }

    @Override
    public void writeTitleTh(ManagerWriter writer) {
        writer.getHtmlWriter().thText(title());
    }
}
