package ru.yandex.solomon.staffOnly.manager.table;

import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.function.Function;

import ru.yandex.solomon.staffOnly.html.CssLine;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.html.HtmlWriterWithCommonLibraries;
import ru.yandex.solomon.staffOnly.manager.ManagerWriter;

/**
 * Writes table header as links to current page with {@code sortBy} parameter.
 *
 * @author Sergey Polovko
 */
public class SortableTableColumnDefImpl<T> implements TableColumnDef<T> {
    private final int index;
    private final String title;
    private final Comparator<T> comparator;
    private final Function<T, ?> content;
    private final List<HtmlWriter.Attr> tdAttrs;

    public enum PredefStyle {
        MONOSPACE(List.of(HtmlWriter.Attr.style(CssLine.fontFamilyMonospace()))),
        ;

        private final List<HtmlWriter.Attr> attrs;

        PredefStyle(List<HtmlWriter.Attr> attrs) {
            this.attrs = attrs;
        }
    }

    public static <T, V extends Comparable<V>>
    SortableTableColumnDefImpl<T> of(int index, String title, Function<T, V> content) {
        return new SortableTableColumnDefImpl<>(index, title, Comparator.comparing(content), content::apply);
    }

    public SortableTableColumnDefImpl(int index, String title, Comparator<T> comparator, Function<T, ?> content) {
        this(index, title, comparator, content, List.of());
    }

    public SortableTableColumnDefImpl(int index, String title, Comparator<T> comparator, Function<T, ?> content, PredefStyle style) {
        this(index, title, comparator, content, style.attrs);
    }

    public SortableTableColumnDefImpl(int index, String title, Comparator<T> comparator, Function<T, ?> content, List<HtmlWriter.Attr> tdAttrs) {
        this.index = index;
        this.title = title;
        this.comparator = comparator;
        this.content = content;
        this.tdAttrs = tdAttrs;
    }

    @Override
    public String title() {
        return title;
    }

    @Override
    public void writeTitleTh(ManagerWriter writer) {
        HtmlWriterWithCommonLibraries html = writer.getHtmlWriter();
        html.th(() -> {
            html.aHref("?sortBy=" + index, title);
        });
    }

    @Override
    public void writeValueTd(T row, ManagerWriter writer) {
        writer.writeCellTdWithAttrs(content.apply(row), tdAttrs);
    }

    @Override
    public void writeValue(T row, ManagerWriter writer) {
        writer.writeCellValue(content.apply(row));
    }

    public void sort(List<T> values, boolean descending) {
        Collections.sort(values, descending ? comparator.reversed() : comparator);
    }

    public void sort(T[] values, boolean descending) {
        Arrays.sort(values, descending ? comparator.reversed() : comparator);
    }
}
